/*
 * Copyright (c) 1998 by David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Most simple built-in commands are here.
 */

#include "sash.h"
#include "extra.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <utime.h>
#include <errno.h>
#include <sys/fs.h>


void
do_echo(argc, argv)
	int		argc;
	const char **	argv;
{
	BOOL	first;

	first = TRUE;

	while (argc-- > 1) {
		if (!first)
			fputc(' ', stdout);

		first = FALSE;
		fputs(*++argv, stdout);
	}

	fputc('\n', stdout);
}


void
do_pwd(argc, argv)
	int		argc;
	const char **	argv;
{
	char	buf[PATHLEN];

	if (getcwd(buf, PATHLEN) == NULL) {
		fprintf(stderr, "Cannot get current directory\n");

		return;
	}

	printf("%s\n", buf);
}


void
do_cd(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	path;

	if (argc > 1)
		path = argv[1];
	else {
		path = getenv("HOME");

		if (path == NULL) {
			fprintf(stderr, "No HOME environment variable\n");

			return;
		}
	}

	if (chdir(path) < 0)
		perror(path);
}


void
do_mkdir(argc, argv)
	int		argc;
	const char **	argv;
{
	while (argc-- > 1) {
		if (mkdir(argv[1], 0777) < 0)
			perror(argv[1]);

		argv++;
	}
}


void
do_mknod(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		mode;
	int		major;
	int		minor;

	mode = 0666;

	if (strcmp(argv[2], "b") == 0)
		mode |= S_IFBLK;
	else if (strcmp(argv[2], "c") == 0)
		mode |= S_IFCHR;
	else {
		fprintf(stderr, "Bad device type\n");

		return;
	}

	major = 0;
	cp = argv[3];

	while (isdecimal(*cp))
		major = major * 10 + *cp++ - '0';

	if (*cp || (major < 0) || (major > 255)) {
		fprintf(stderr, "Bad major number\n");

		return;
	}

	minor = 0;
	cp = argv[4];

	while (isdecimal(*cp))
		minor = minor * 10 + *cp++ - '0';

	if (*cp || (minor < 0) || (minor > 255)) {
		fprintf(stderr, "Bad minor number\n");

		return;
	}

	if (mknod(argv[1], mode, major * 256 + minor) < 0)
		perror(argv[1]);
}


void
do_rmdir(argc, argv)
	int		argc;
	const char **	argv;
{
	while (argc-- > 1) {
		if (rmdir(argv[1]) < 0)
			perror(argv[1]);

		argv++;
	}
}


void
do_sync(argc, argv)
	int		argc;
	const char **	argv;
{
	sync();
}


void
do_rm(argc, argv)
	int		argc;
	const char **	argv;
{
	while (argc-- > 1) {
		if (unlink(argv[1]) < 0)
			perror(argv[1]);

		argv++;
	}
}


void
do_chmod(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		mode;

	mode = 0;
	cp = argv[1];

	while (isoctal(*cp))
		mode = mode * 8 + (*cp++ - '0');

	if (*cp) {
		fprintf(stderr, "Mode must be octal\n");

		return;
	}

	argc--;
	argv++;

	while (argc-- > 1) {
		if (chmod(argv[1], mode) < 0)
			perror(argv[1]);

		argv++;
	}
}


void
do_chown(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		uid;
	struct passwd *	pwd;
	struct stat	statbuf;

	cp = argv[1];

	if (isdecimal(*cp)) {
		uid = 0;

		while (isdecimal(*cp))
			uid = uid * 10 + (*cp++ - '0');

		if (*cp) {
			fprintf(stderr, "Bad uid value\n");

			return;
		}
	} else {
		pwd = getpwnam(cp);

		if (pwd == NULL) {
			fprintf(stderr, "Unknown user name\n");

			return;
		}

		uid = pwd->pw_uid;
	}

	argc--;
	argv++;

	while (argc-- > 1) {
		argv++;

		if ((stat(*argv, &statbuf) < 0) ||
			(chown(*argv, uid, statbuf.st_gid) < 0))
		{
			perror(*argv);
		}
	}
}


void
do_chgrp(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		gid;
	struct group *	grp;
	struct stat	statbuf;

	cp = argv[1];

	if (isdecimal(*cp)) {
		gid = 0;

		while (isdecimal(*cp))
			gid = gid * 10 + (*cp++ - '0');

		if (*cp) {
			fprintf(stderr, "Bad gid value\n");

			return;
		}
	} else {
		grp = getgrnam(cp);

		if (grp == NULL) {
			fprintf(stderr, "Unknown group name\n");

			return;
		}

		gid = grp->gr_gid;
	}

	argc--;
	argv++;

	while (argc-- > 1) {
		argv++;

		if ((stat(*argv, &statbuf) < 0) ||
			(chown(*argv, statbuf.st_uid, gid) < 0))
		{
			perror(*argv);
		}
	}
}


void
do_touch(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	name;
	int		fd;
	struct utimbuf	now;

	time(&now.actime);
	now.modtime = now.actime;

	while (argc-- > 1) {
		name = *(++argv);

		fd = open(name, O_CREAT | O_WRONLY | O_EXCL, 0666);

		if (fd >= 0) {
			close(fd);

			continue;
		}

		if (utime(name, &now) < 0)
			perror(name);
	}
}


void
do_mv(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	srcname;
	const char *	destname;
	const char *	lastarg;
	BOOL		dirflag;

	lastarg = argv[argc - 1];

	dirflag = isadir(lastarg);

	if ((argc > 3) && !dirflag) {
		fprintf(stderr, "%s: not a directory\n", lastarg);

		return;
	}

	while (!intflag && (argc-- > 2)) {
		srcname = *(++argv);

		if (access(srcname, 0) < 0) {
			perror(srcname);

			continue;
		}

		destname = lastarg;

		if (dirflag)
			destname = buildname(destname, srcname);

		if (rename(srcname, destname) >= 0)
			continue;

		if (errno != EXDEV) {
			perror(destname);

			continue;
		}

		if (!copyfile(srcname, destname, TRUE))
			continue;

		if (unlink(srcname) < 0)
			perror(srcname);
	}
}


void
do_ln(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	srcname;
	const char *	destname;
	const char *	lastarg;
	BOOL		dirflag;

	if (argv[1][0] == '-') {
		if (strcmp(argv[1], "-s")) {
			fprintf(stderr, "Unknown option\n");

			return;
		}

		if (argc != 4) {
			fprintf(stderr, "Wrong number of arguments for symbolic link\n");

			return;
		}

#ifdef	S_ISLNK
		if (symlink(argv[2], argv[3]) < 0)
			perror(argv[3]);
#else
		fprintf(stderr, "Symbolic links are not allowed\n");
#endif
		return;
	}

	/*
	 * Here for normal hard links.
	 */
	lastarg = argv[argc - 1];
	dirflag = isadir(lastarg);

	if ((argc > 3) && !dirflag) {
		fprintf(stderr, "%s: not a directory\n", lastarg);

		return;
	}

	while (argc-- > 2) {
		srcname = *(++argv);

		if (access(srcname, 0) < 0) {
			perror(srcname);

			continue;
		}

		destname = lastarg;

		if (dirflag)
			destname = buildname(destname, srcname);

		if (link(srcname, destname) < 0) {
			perror(destname);

			continue;
		}
	}
}


void
do_cp(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	srcname;
	const char *	destname;
	const char *	lastarg;
	BOOL		dirflag;

	lastarg = argv[argc - 1];

	dirflag = isadir(lastarg);

	if ((argc > 3) && !dirflag) {
		fprintf(stderr, "%s: not a directory\n", lastarg);

		return;
	}

	while (!intflag && (argc-- > 2)) {
		srcname = *argv++;
		destname = lastarg;

		if (dirflag)
			destname = buildname(destname, srcname);

		(void) copyfile(srcname, destname, FALSE);
	}
}


void
do_mount(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	str;
	const char *	type;
	int		flags;

	argc--;
	argv++;
	type = "ext2";
	flags = MS_MGC_VAL;

	while ((argc > 0) && (**argv == '-')) {
		argc--;
		str = *argv++ ;

		while (*++str) switch (*str) {
			case 't':
				if ((argc <= 0) || (**argv == '-')) {
					fprintf(stderr, "Missing file system type\n");

					return;
				}

				type = *argv++;
				argc--;
				break;

			case 'r':
				flags |= MS_RDONLY;
				break;

			case 'm':
				flags |= MS_REMOUNT;
				break;

			default:
				fprintf(stderr, "Unknown option\n");

				return;
		}
	}

	if (argc != 2) {
		fprintf(stderr, "Wrong number of arguments for mount\n");

		return;
	}

	if (mount(argv[0], argv[1], type, flags, 0) < 0)
		perror("mount failed");
}


void
do_umount(argc, argv)
	int		argc;
	const char **	argv;
{
	if (umount(argv[1]) < 0)
		perror(argv[1]);
}


void
do_cmp(argc, argv)
	int		argc;
	const char **	argv;
{
	int		fd1;
	int		fd2;
	int		cc1;
	int		cc2;
	long		pos;
	const char *	bp1;
	const char *	bp2;
	char		buf1[BUFSIZE];
	char		buf2[BUFSIZE];
	struct	stat	statbuf1;
	struct	stat	statbuf2;

	if (stat(argv[1], &statbuf1) < 0) {
		perror(argv[1]);

		return;
	}

	if (stat(argv[2], &statbuf2) < 0) {
		perror(argv[2]);

		return;
	}

	if ((statbuf1.st_dev == statbuf2.st_dev) &&
		(statbuf1.st_ino == statbuf2.st_ino))
	{
		printf("Files are links to each other\n");

		return;
	}

	if (statbuf1.st_size != statbuf2.st_size) {
		printf("Files are different sizes\n");

		return;
	}

	fd1 = open(argv[1], O_RDONLY);

	if (fd1 < 0) {
		perror(argv[1]);

		return;
	}

	fd2 = open(argv[2], O_RDONLY);

	if (fd2 < 0) {
		perror(argv[2]);
		close(fd1);

		return;
	}

	pos = 0;

	while (TRUE) {
		if (intflag)
			goto closefiles;

		cc1 = read(fd1, buf1, sizeof(buf1));

		if (cc1 < 0) {
			perror(argv[1]);
			goto closefiles;
		}

		cc2 = read(fd2, buf2, sizeof(buf2));

		if (cc2 < 0) {
			perror(argv[2]);
			goto closefiles;
		}

		if ((cc1 == 0) && (cc2 == 0)) {
			printf("Files are identical\n");
			goto closefiles;
		}

		if (cc1 < cc2) {
			printf("First file is shorter than second\n");
			goto closefiles;
		}

		if (cc1 > cc2) {
			printf("Second file is shorter than first\n");
			goto closefiles;
		}

		if (memcmp(buf1, buf2, cc1) == 0) {
			pos += cc1;

			continue;
		}

		bp1 = buf1;
		bp2 = buf2;
		while (*bp1++ == *bp2++)
			pos++;

		printf("Files differ at byte position %ld\n", pos);

		goto closefiles;
	}

closefiles:
	close(fd1);
	close(fd2);
}


void
do_more(argc, argv)
	int		argc;
	const char **	argv;
{
	FILE *		fp;
	const char *	name;
	int		ch;
	int		line;
	int		col;
	int		pageLines;
	int		pageColumns;
	char		buf[80];

	/*
	 * Get the width and height of the screen if it is set.
	 * If not, then default it.
	 */
	pageLines = 0;
	pageColumns = 0;

	name = getenv("LINES");

	if (name)
		pageLines = atoi(name);

	name = getenv("COLS");

	if (name)
		pageColumns = atoi(name);

	if (pageLines <= 0)
		pageLines = 24;

	if (pageColumns <= 0)
		pageColumns = 80;

	/*
	 * OK, process each file.
	 */
	while (argc-- > 1) {
		name = *(++argv);

		fp = fopen(name, "r");

		if (fp == NULL) {
			perror(name);

			return;
		}

		printf("<< %s >>\n", name);
		line = 1;
		col = 0;

		while (fp && ((ch = fgetc(fp)) != EOF)) {
			switch (ch) {
				case '\r':
					col = 0;
					break;

				case '\n':
					line++;
					col = 0;
					break;

				case '\t':
					col = ((col + 1) | 0x07) + 1;
					break;

				case '\b':
					if (col > 0)
						col--;
					break;

				default:
					col++;
			}

			putchar(ch);

			if (col >= pageColumns) {
				col -= pageColumns;
				line++;
			}

			if (line < pageLines)
				continue;

			if (col > 0)
				putchar('\n');

			printf("--More--");
			fflush(stdout);

			if (intflag || (read(0, buf, sizeof(buf)) < 0)) {
				if (fp)
					fclose(fp);

				return;
			}

			ch = buf[0];

			if (ch == ':')
				ch = buf[1];

			switch (ch) {
				case 'N':
				case 'n':
					fclose(fp);
					fp = NULL;
					break;

				case 'Q':
				case 'q':
					fclose(fp);

					return;
			}

			col = 0;
			line = 1;
		}

		if (fp)
			fclose(fp);
	}
}


void
do_exit(argc, argv)
	int		argc;
	const char **	argv;
{
	exit(0);
}


void
do_setenv(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	name;
	const char *	value;
	char *		str;

	name = argv[1];
	value = argv[2];

	/*
	 * The value given to putenv must remain around, so we must malloc it.
	 * Note: memory is not reclaimed if the same variable is redefined.
	 */
	str = malloc(strlen(name) + strlen(value) + 2);

	if (str == NULL)
	{
		fprintf(stderr, "Cannot allocate memory\n");

		return;
	}

	strcpy(str, name);
	strcat(str, "=");
	strcat(str, value);

	putenv(str);
}


void
do_printenv(argc, argv)
	int		argc;
	const char **	argv;
{
	const char **	env;
	extern char **	environ;
	int		len;

	env = (const char **) environ;

	if (argc == 1) {
		while (*env)
			printf("%s\n", *env++);

		return;
	}

	len = strlen(argv[1]);

	while (*env) {
		if ((strlen(*env) > len) && (env[0][len] == '=') &&
			(memcmp(argv[1], *env, len) == 0))
		{
			printf("%s\n", &env[0][len+1]);

			return;
		}
		env++;
	}
}


void
do_umask(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		mask;

	if (argc <= 1) {
		mask = umask(0);
		umask(mask);
		printf("%03o\n", mask);

		return;
	}

	mask = 0;
	cp = argv[1];

	while (isoctal(*cp))
		mask = mask * 8 + *cp++ - '0';

	if (*cp || (mask & ~0777)) {
		fprintf(stderr, "Bad umask value\n");

		return;
	}

	umask(mask);
}


void
do_kill(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	cp;
	int		sig;
	int		pid;

	sig = SIGTERM;

	if (argv[1][0] == '-') {
		cp = &argv[1][1];

		if (strcmp(cp, "HUP") == 0)
			sig = SIGHUP;
		else if (strcmp(cp, "INT") == 0)
			sig = SIGINT;
		else if (strcmp(cp, "QUIT") == 0)
			sig = SIGQUIT;
		else if (strcmp(cp, "KILL") == 0)
			sig = SIGKILL;
		else {
			sig = 0;

			while (isdecimal(*cp))
				sig = sig * 10 + *cp++ - '0';

			if (*cp) {
				fprintf(stderr, "Unknown signal\n");

				return;
			}
		}

		argc--;
		argv++;
	}

	while (argc-- > 1) {
		cp = *++argv;
		pid = 0;

		while (isdecimal(*cp))
			pid = pid * 10 + *cp++ - '0';

		if (*cp) {
			fprintf(stderr, "Non-numeric pid\n");

			return;
		}

		if (kill(pid, sig) < 0)
			perror(*argv);
	}
}


void
do_where(argc, argv)
	int		argc;
	const char **	argv;
{
	const char *	program;
	const char *	dirName;
	char *		path;
	char *		endPath;
	char *		fullPath;
	BOOL		found;

	found = FALSE;
	program = argv[1];

	if (strchr(program, '/') != NULL)
	{
		fprintf(stderr, "Program name cannot include a path\n");

		return;
	}

	path = getenv("PATH");

	fullPath = getchunk(strlen(path) + strlen(program) + 2);
	path = chunkstrdup(path);

	if ((path == NULL) || (fullPath == NULL))
	{
		fprintf(stderr, "Memory allocation failed\n");

		return;
	}

	/*
	 * Check out each path to see if the program exists and is
	 * executable in that path.
	 */
	for (; path; path = endPath)
	{
		/*
		 * Find the end of the next path and NULL terminate
		 * it if necessary.
		 */
		endPath = strchr(path, ':');

		if (endPath)
			*endPath++ = '\0';

		/*
		 * Get the directory name, defaulting it to DOT if
		 * it is null.
		 */
		dirName = path;

		if (dirName == '\0')
			dirName = ".";

		/*
		 * Construct the full path of the program.
		 */
		strcpy(fullPath, dirName);
		strcat(fullPath, "/");
		strcat(fullPath, program);

		/*
		 * See if the program exists and is executable.
		 */
		if (access(fullPath, X_OK) < 0)
		{
			if (errno != ENOENT)
				printf("%s: %s\n", fullPath, strerror(errno));

			continue;
		}

		printf("%s\n", fullPath);
		found = TRUE;
	}

	if (!found)
		printf("Program \"%s\" not found in PATH\n", program);
}

/* END CODE */
