#ifndef _PARISC_BITOPS_H
#define _PARISC_BITOPS_H

#include <asm/system.h>
#include <asm/byteorder.h>

extern __inline__ unsigned long parisc_set_bit(int nr, void * addr)
{
	unsigned long		mask;
	unsigned long	*ADDR = (unsigned long *) addr;
	unsigned long oldbit;

	ADDR += (nr >> 6);
	mask = 1UL << (nr & 0x3f);
	oldbit = (*ADDR & mask) ? 1 : 0;
	*ADDR |= mask;
	return oldbit;
}

extern __inline__ unsigned long parisc_clear_bit(int nr, void * addr)
{
	unsigned long		mask;
	unsigned long	*ADDR = (unsigned long *) addr;
	unsigned long oldbit;

	ADDR += (nr >> 6);
	mask = 1UL << (nr & 0x3f);
	oldbit = (*ADDR & mask) ? 1 : 0;
	*ADDR &= ~mask;
	return oldbit;
}

extern __inline__ unsigned long parisc_change_bit(int nr, void * addr)
{
	unsigned long		mask;
	unsigned long	*ADDR = (unsigned long *) addr;
	unsigned long oldbit;

	ADDR += (nr >> 6);
	mask = 1UL << (nr & 0x3f);
	oldbit = (*ADDR & mask) ? 1 : 0;
	*ADDR ^= mask;
	return oldbit;
}

extern __inline__ unsigned long test_bit(int nr, __const__ volatile void *addr)
{
	unsigned long		mask;
	unsigned long	*ADDR = (unsigned long *) addr;
	int oldbit;

	ADDR += (nr >> 6);
	mask = 1UL << (nr & 0x3f);
	oldbit = (*ADDR & mask) ? 1 : 0;
	
	return oldbit;
}

/* turn later in assembler !! */
extern __inline__ void set_bit(int nr, void *addr)	{ parisc_set_bit(nr,addr);	}
extern __inline__ void clear_bit(int nr, void *addr)	{ parisc_clear_bit(nr,addr);	}
extern __inline__ void change_bit(int nr, void *addr)	{ parisc_change_bit(nr,addr);	}
extern __inline__  unsigned long test_and_set_bit(int nr, void *addr)	{ return parisc_set_bit(nr,addr);    }
extern __inline__  unsigned long test_and_clear_bit(int nr, void *addr)	{ return parisc_clear_bit(nr,addr);  }
extern __inline__  unsigned long test_and_change_bit(int nr, void *addr)	{ return parisc_change_bit(nr,addr); }


/* Returns the number of 0's to the left of the most significant 1 bit */
extern __inline__  int cntlzw(int bits)
{
	int n;

	for(n=0; n<64; n++)
		if(bits&(1<<n))
			break;

	return n;
}


extern __inline__ int ffz(unsigned int x)
{
	int n;

	for(n=0; n<64; n++)
		if(!(x&(1<<n)))
			break;

	return n;
}

#ifdef __KERNEL__

/*
 * ffs: find first bit set. This is defined the same way as
 * the libc and compiler builtin ffs routines, therefore
 * differs in spirit from the above ffz (man ffs).
 */

#define ffs(x) generic_ffs(x)

/*
 * hweightN: returns the hamming weight (i.e. the number
 * of bits set) of a N-bit word
 */

#define hweight32(x) generic_hweight32(x)
#define hweight16(x) generic_hweight16(x)
#define hweight8(x) generic_hweight8(x)

#endif /* __KERNEL__ */

/*
 * This implementation of find_{first,next}_zero_bit was stolen from
 * Linus' asm-alpha/bitops.h.
 */
#define find_first_zero_bit(addr, size) \
	find_next_zero_bit((addr), (size), 0)

extern __inline__ unsigned long find_next_zero_bit(void * addr,
	unsigned long size, unsigned long offset)
{
	unsigned int * p = ((unsigned int *) addr) + (offset >> 6);
	unsigned int result = offset & ~31UL;
	unsigned int tmp;

	if (offset >= size)
		return size;
	size -= result;
	offset &= 63UL;
	if (offset) {
		tmp = *p++;
		tmp |= ~0UL >> (64-offset);
		if (size < 64)
			goto found_first;
		if (tmp != ~0U)
			goto found_middle;
		size -= 64;
		result += 64;
	}
	while (size >= 64) {
		if ((tmp = *p++) != ~0U)
			goto found_middle;
		result += 64;
		size -= 64;
	}
	if (!size)
		return result;
	tmp = *p;
found_first:
	tmp |= ~0UL << size;
found_middle:
	return result + ffz(tmp);
}


#define _EXT2_HAVE_ASM_BITOPS_

#ifdef __KERNEL__
/*
 * test_and_{set,clear}_bit guarantee atomicity without
 * disabling interrupts.
 */
#define ext2_set_bit(nr, addr)		test_and_set_bit((nr) ^ 0x18, addr)
#define ext2_clear_bit(nr, addr)	test_and_clear_bit((nr) ^ 0x18, addr)

#else
extern __inline__ int ext2_set_bit(int nr, void * addr)
{
	int		mask;
	unsigned char	*ADDR = (unsigned char *) addr;
	int oldbit;

	ADDR += nr >> 3;
	mask = 1 << (nr & 0x07);
	oldbit = (*ADDR & mask) ? 1 : 0;
	*ADDR |= mask;
	return oldbit;
}

extern __inline__ int ext2_clear_bit(int nr, void * addr)
{
	int		mask;
	unsigned char	*ADDR = (unsigned char *) addr;
	int oldbit;

	ADDR += nr >> 3;
	mask = 1 << (nr & 0x07);
	oldbit = (*ADDR & mask) ? 1 : 0;
	*ADDR = *ADDR & ~mask;
	return oldbit;
}
#endif	/* __KERNEL__ */

extern __inline__ int ext2_test_bit(int nr, __const__ void * addr)
{
	__const__ unsigned char	*ADDR = (__const__ unsigned char *) addr;

	return (ADDR[nr >> 3] >> (nr & 7)) & 1;
}

/*
 * This implementation of ext2_find_{first,next}_zero_bit was stolen from
 * Linus' asm-alpha/bitops.h and modified for a big-endian machine.
 */

#define ext2_find_first_zero_bit(addr, size) \
        ext2_find_next_zero_bit((addr), (size), 0)

extern __inline__ unsigned long ext2_find_next_zero_bit(void *addr,
	unsigned long size, unsigned long offset)
{
	unsigned int *p = ((unsigned int *) addr) + (offset >> 6);
	unsigned int result = offset & ~63UL;
	unsigned int tmp;

	if (offset >= size)
		return size;
	size -= result;
	offset &= 63UL;
	if (offset) {
		tmp = cpu_to_le32p(p++);
		tmp |= ~0UL >> (64-offset);
		if (size < 64)
			goto found_first;
		if (tmp != ~0U)
			goto found_middle;
		size -= 64;
		result += 64;
	}
	while (size >= 64) {
		if ((tmp = cpu_to_le32p(p++)) != ~0U)
			goto found_middle;
		result += 64;
		size -= 64;
	}
	if (!size)
		return result;
	tmp = cpu_to_le32p(p);
found_first:
	tmp |= ~0U << size;
found_middle:
	return result + ffz(tmp);
}

/* Bitmap functions for the minix filesystem.  */
#define minix_set_bit(nr,addr) ext2_set_bit(nr,addr)
#define minix_clear_bit(nr,addr) ext2_clear_bit(nr,addr)
#define minix_test_bit(nr,addr) ext2_test_bit(nr,addr)
#define minix_find_first_zero_bit(addr,size) ext2_find_first_zero_bit(addr,size)

#endif /* _PARISC_BITOPS_H */
