/******************************************************/
/* iodc.c: main code for IODC Block driver.           */
/*                                                    */
/* leveraged fomr Linux Device Driver book.           */
/*                                                    */
/******************************************************/

#define DEVICE_OFF(d) /* do-nothing */
#define DEVICE_NO_RANDOM 1

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/genhd.h>
#include <linux/delay.h>
#include <linux/mm.h>
#include <linux/major.h>
#define MAJOR_NR UNNAMED_MAJOR
#include <linux/blk.h>
#include <linux/malloc.h>
#include <linux/interrupt.h>

#include <linux/hdreg.h> /* for HDIO_GETGEO */

#include <asm/setup.h>
#include <asm/pgtable.h>
#include <asm/system.h>
#include <asm/uaccess.h>

#include <asm/iodc.h>

#define panic printk

extern struct file_operations iodc_fops;
#include <asm/hack.h>
funcDescrPtr IodcFuncDptr;
 
#define  ROUND_UP(x,y)       ((((x) + ((y) - 1)) / (y)) * (y))
#define MAJOR_NR iodc_major /* force definitions on in blk.h */
static int iodc_major    = IODC_MAJOR;

#define DEVICE_NR(device) MINOR(device)   /* iodc has no partition bits */
#define DEVICE_NAME "iodc"                /* name for messaging */
#define DEVICE_INTR iodc_intrptr          /* pointer to the bottom half */
#define DEVICE_NO_RANDOM                  /* no entropy to contribute */



static int iodc_devs     = IODC_DEVS;
static int iodc_rahead   = IODC_RAHEAD;
static int iodc_size     = IODC_SIZE;
static int iodc_blksize  = IODC_BLKSIZE;
static int iodc_hardsect = IODC_HARDSECT;

/* this structure will hold relavent information for the iodc driver */
static IODC_INFO tracker;

/********* IODC File Operations Procedures ************/

int iodc_open (struct inode *inode, struct file *filp)
{
	MOD_INC_USE_COUNT;
	return 0;
}

void iodc_release (struct inode *inode, struct file *filp)
{
	fsync_dev(inode->i_rdev);
	invalidate_buffers(inode->i_rdev);

	MOD_DEC_USE_COUNT;
}

int iodc_ioctl (struct inode *inode, struct file *filp,
                 unsigned int cmd, unsigned long arg)
{
	unsigned long rret[32];	/* return array */
   	int	file_start = inode->u.lif_i.file_start * inode->i_blksize;
        int	file_size  = inode->i_blocks * inode->i_blksize; /* in bytes */
        char	*buf = filp->private_data;
	int 	ret, vret, block_count;
	unsigned long crs[8];	/* return array */

	switch(cmd) {

        /* Reading a LIF file */
        case IODC_READ_FILE:

            /* The most important assumption here is that the files on LIF
             * are aligned on 2K boundaries since that is how it is created
             * currently. */
	    if(tracker.bdev_type == CL_SEQU) { /* sequential read */
		
		/* see if we need to sequence upto requested addr */
		if(tracker.seq_addr != file_start) {

	          /* if we have not reached desired addr yet */
		  if(tracker.seq_addr > file_start) {

	              /* nead semaphore lock here */

	              vret = enter_real();

		      SAVE_CRS(crs)

	              /* call IODC directly in real mode: */
		      /* REWIND */
	              /* Note: 1-to-1 virtual data mapping for kernel */
	              /* making assumptions about sector meaning, etc. */
	              ret = (*tracker.bdev_iodc_entry)(tracker.bdev_HPA, 0, 
                           tracker.bdev_SPA, tracker.bdev_layers, rret, 
		           0, PA(buf), file_size, 0); /* need buf, file_size ?? */

                      RESTORE_CRS(crs)

	              leave_real(vret);

	      	      /* nead semaphore lock here */

		      if ( ret < 0 )
                      {
			 printk("\nret from iodc driver = %d", ret);
	                 return ret;
                      }

		      tracker.seq_addr = 0;
		    
		  } /* end if we have not reached desired addr yet */

		  /* Calculate the # of 2k blocks we need to read to */
		  /* get to the requested address */
		  block_count = 
		    (file_start - tracker.seq_addr) / CONST_2K;

		  while(block_count)
		  {
			   /* nead semaphore lock here */

			   vret = enter_real();

                           /* Control registers need to be saved and restored */
			   SAVE_CRS(crs)

	      		   /* call IODC directly in real mode: */
			   /* Read 2k chunks at a time to get to req addr */
	      		   /* Note: 1-to-1 virtual data mapping for kernel */
	      		   /* making assumptions about sector meaning, etc. */
	      		   ret = (*tracker.bdev_iodc_entry)(tracker.bdev_HPA, 
				0, tracker.bdev_SPA, tracker.bdev_layers, rret, 
		   	        tracker.seq_addr, PA(buf), 
		   	      CONST_2K, 0);

			   RESTORE_CRS(crs)

	      		   leave_real(vret);

			   if(ret >= 0) {
			     tracker.seq_addr = tracker.seq_addr + rret[0];
			   }
			   else {
			    printk("\nIodc driver in block cont loop ret = %d",ret);
			    return ret;
			   }

			   block_count = block_count - CONST_2K;
		  } /* end while */

		} /* end if we need to sequence upto requested addr */ 

		/* at this point, the dev addr is the next one to grab */
	    }

	    /* At this point, we are doing either 1 contiguous random */
	    /* access read, or we have sequenced up to the point to do */
	    /* a sequential contiguous read: */

	    /* nead semaphore lock here */

	    vret = enter_real();

            SAVE_CRS(crs)

	    printk("\n args: file_start = %x buf = %x size = %x \n",
		file_start, PA(buf), file_size);
	    /* call IODC directly in real mode: */
	    /* Note: 1-to-1 virtual data mapping for kernel */
	    /* making assumptions about sector meaning, etc. */
	    ret = (*tracker.bdev_iodc_entry)(tracker.bdev_HPA, 0, 
                 tracker.bdev_SPA, tracker.bdev_layers, rret, 
		 file_start, PA(buf), ROUND_UP(file_size,2048), ROUND_UP(file_size,2048));

	    RESTORE_CRS(crs)

	    leave_real(vret);

	    /* nead semaphore lock here */
	     
	   printk("\nFinal IODC driver read ret = %d",ret);
	   return ret; 

	  case BLKGETSIZE:
        /* Return the device size, expressed in sectors */
	return -EINVAL;	

	case BLKFLSBUF: /* flush */
	fsync_dev(inode->i_rdev);
        invalidate_buffers(inode->i_rdev);
        return 0;

	case BLKRAGET: /* return the readahead value */
	return -EINVAL;

	case BLKRASET: /* set the readahead value */
	return -EINVAL;

	case BLKRRPART: /* re-read partition table: can't do it */
        return -EINVAL;

	case HDIO_GETGEO: /* get geometry: */
	} /* end switch */

	return -EINVAL; /* unknown command */
}

/* Main stuf hapens here: */
/* making assumptions about sector meaning, etc. */
/* I will assume that a sector equals the boot device address */
void iodc_request(void)
{
	unsigned long rret[32];	/* return array */
	int ret, vret, block_count;
	struct buffer_head *bhead;
	unsigned long crs[8];	/* return array */

	INIT_REQUEST;		/* validate request */

	switch(CURRENT->cmd) {
          case READ: /*--------------------------------------*/

	    bhead = CURRENT->bh; /* may not need this */
		
	    if(tracker.bdev_type == CL_SEQU) { /* sequential read */
		
		/* see if we need to sequence upto requested addr */
		if(tracker.seq_addr != (int)CURRENT->sector) {

	          /* if we have not reached desired addr yet */
		  if(tracker.seq_addr > (int)CURRENT->sector) {

	              /* nead semaphore lock here */

	              vret = enter_real();

                      SAVE_CRS(crs)

	              /* call IODC directly in real mode: */
		      /* REWIND */
	              /* Note: 1-to-1 virtual data mapping for kernel */
	              /* making assumptions about sector meaning, etc. */
	              ret = (*tracker.bdev_iodc_entry)(tracker.bdev_HPA, 0, 
                           tracker.bdev_SPA, tracker.bdev_layers, rret, 
		           0, CURRENT->buffer, 
		           (CURRENT->current_nr_sectors * iodc_hardsect), 0);

		      RESTORE_CRS(crs)

	              leave_real(vret);

	      	      /* nead semaphore lock here */

	              if(ret >= 0)
		        end_request(1); /* success */
	              else
		         end_request(0);

		      tracker.seq_addr = 0;
		    
		  } /* end if we have not reached desired addr yet */

		  /* Calculate the # of 2k blocks we need to read to */
		  /* get to the requested address */
		  block_count = 
		    ((int)CURRENT->sector - tracker.seq_addr) / CONST_2K;

		  while(block_count)
		  {
			   /* nead semaphore lock here */

			   vret = enter_real();

			   SAVE_CRS(crs)
	      		   /* call IODC directly in real mode: */
			   /* Read 2k chunks at a time to get to req addr */
	      		   /* Note: 1-to-1 virtual data mapping for kernel */
	      		   /* making assumptions about sector meaning, etc. */
	      		   ret = (*tracker.bdev_iodc_entry)(tracker.bdev_HPA, 
				0, tracker.bdev_SPA, tracker.bdev_layers, rret, 
		   	        tracker.seq_addr, CURRENT->buffer, 
		   	      CONST_2K, 0);
			   RESTORE_CRS(crs)
	      		   leave_real(vret);

			   if(ret >= 0) {
			     tracker.seq_addr = tracker.seq_addr + rret[0];
			   }
			   else
			    end_request(0); /* failure */

			   block_count = block_count - CONST_2K;
		  } /* end while */

		} /* end if we need to sequence upto requested addr */ 

		/* at this point, the dev addr is the next one to grab */
	    }

	    /* At this point, we are doing either 1 contiguous random */
	    /* access read, or we have sequenced up to the point to do */
	    /* a sequential contiguous read: */

	    /* nead semaphore lock here */

	    vret = enter_real();

	    SAVE_CRS(crs)

	    /* call IODC directly in real mode: */
	    /* Note: 1-to-1 virtual data mapping for kernel */
	    /* making assumptions about sector meaning, etc. */
	    ret = (*tracker.bdev_iodc_entry)(tracker.bdev_HPA, 0, 
                 tracker.bdev_SPA, tracker.bdev_layers, rret, 
		 (int)CURRENT->sector, CURRENT->buffer, 
		 (CURRENT->current_nr_sectors * iodc_hardsect), 0);

            RESTORE_CRS(crs)
	    leave_real(vret);

	    /* nead semaphore lock here */

	    if(ret >= 0)
		end_request(1); /* success */
	    else
		 end_request(0);


            break;

          case WRITE: /*--------------------------------------*/
          default: /*--------------------------------------*/
            /* can't happen */
            end_request(0);
            
        } /* end switch */

        end_request(1); /* success */
}


/*****************************************/
/* Initialize the iodc driver interface.
/*****************************************/
int iodc_init(void)
{
	unsigned int *ptr;      /*generic 32 bit ptr */
	int ret,rret,vret;
	unsigned long aret[32];
	unsigned long crs[8];
	char bdevbuf[2048] __attribute__((aligned(8)));

	/* find & grab iodc entry */
        ptr = (unsigned *)(PG0_BOOT_DEVICE + PG0_OFF_IODC_IO);

	/* NEED Function Pointer Hack here : */
	IodcFuncDptr.funcAddr = (unsigned long)*ptr;
	asm("copy 27,%0" : "=r" (IodcFuncDptr.gpVal));
	tracker.bdev_iodc_entry = (unsigned int (*)())&IodcFuncDptr;

	/* find & grab boot device HPA */
        ptr = (unsigned *)(PG0_BOOT_DEVICE + PG0_OFF_HPA);
        tracker.bdev_HPA = *ptr;

	/* find & grab boot device SPA */
        ptr = (unsigned *)(PG0_BOOT_DEVICE + PG0_OFF_SPA);
        tracker.bdev_SPA = *ptr;

        /* find & grab layers */
        tracker.bdev_layers = (unsigned *)(PG0_BOOT_DEVICE + PG0_OFF_LAYERS);
        tracker.bdev_type = (unsigned *)(PG0_BOOT_DEVICE + PG0_OFF_CLASS);
        tracker.bdev_type &= 0x0F;

        if(tracker.bdev_type == CL_SEQU) {  /* rewind */
	  vret = enter_real();
	  SAVE_CRS(crs)
	  /* call iodc to rewind */
	  ret = 
	    (*tracker.bdev_iodc_entry)(tracker.bdev_HPA, 0, tracker.bdev_SPA,
		tracker.bdev_layers, aret, 0, bdevbuf, sizeof(bdevbuf), 0);
          RESTORE_CRS(crs)
	  leave_real(vret);
	  tracker.seq_addr = 0; /* boot dev now rewound */
	}

	/* register driver */
	if( register_blkdev( IODC_MAJOR, "hp_iodc", &iodc_fops) )
	{
	    printk(KERN_ERR "hp_iodc: Unable to get major %d\n",IODC_MAJOR);
	    return(-ENXIO);
	}

	blk_dev[IODC_MAJOR].request_fn = iodc_request;
	blk_size[IODC_MAJOR] = NULL;	 /* don't check device capacity */
	blksize_size[IODC_MAJOR] = iodc_blksize; /* iodc has 2k block size */
	hardsect_size[IODC_MAJOR] = iodc_hardsect; /* do we use this? */
	read_ahead[IODC_MAJOR] = iodc_rahead; /* don't read ahead */
	
	return(0);
}

/********* IODC File Operations ************/

struct file_operations iodc_fops = {
    NULL,          /* lseek: default */
    block_read,	   /* read - general block-dev read */
    block_write,   /* write - general block-dev write */
    NULL,          /* readdir */
    NULL,          /* select */
    iodc_ioctl,
    NULL,          /* mmap */
    iodc_open,
    iodc_release,
    block_fsync,
    NULL,          /* fasync */
    NULL,	   /* check_change */
    NULL	   /* revalidate */
};
