#ifndef __PARISC_SPINLOCK_T_H
#define __PARISC_SPINLOCK_T_H

/* LDCW, the only atomic read-write operation PA-RISC has. *sigh*.
 *
 * Note that PA-RISC has to use `1' to mean unlocked and `0' to mean locked
 * since it only has load-and-zero.
 */
#define __ldcw(a) ({ \
	unsigned __ret; \
	__asm__ __volatile__("ldcw,ma 0(%1),%0" \
                             : "=r" (__ret) : "r" (a) : "memory"); \
	__ret; \
})

/*
 * Your basic SMP spinlocks, allowing only a single CPU anywhere
 */

typedef struct {
#ifdef CONFIG_PA20
	volatile unsigned int lock;
#else
	volatile unsigned int __attribute__((aligned(16))) lock;
#endif
#ifdef CONFIG_DEBUG_SPINLOCK
	volatile unsigned long owner_pc;
	volatile unsigned long owner_cpu;
#endif
} spinlock_t;

#ifndef CONFIG_DEBUG_SPINLOCK
#define SPIN_LOCK_UNLOCKED_INIT { 1 }
#define SPIN_LOCK_UNLOCKED (spinlock_t) SPIN_LOCK_UNLOCKED_INIT

/* Define 6 spinlock primitives that don't depend on anything else. */

#define spin_lock_init(x)       do { (x)->lock = 1; } while(0)
#define spin_is_locked(x)       ((x)->lock == 0)
#define spin_trylock(x)		(__ldcw(&(x)->lock) != 0)
 
/* 
 * PA2.0 is not strongly ordered.  PA1.X is strongly ordered.
 * ldcw enforces ordering and we need to make sure ordering is
 * enforced on the unlock too.
 * "stw,ma" with Zero index is an alias for "stw,o".
 * But PA 1.x can assemble the "stw,ma" while it doesn't know about "stw,o".
 * And PA 2.0 will generate the right insn using either form.
 * Thanks to John David Anglin for this cute trick.
 *
 * Writing this with asm also ensures that the unlock doesn't
 * get reordered
 */
#define spin_unlock(x) do { __asm__ __volatile__ ("stw,ma  %%sp,0(%0)" \
                                    : : "r" (&(x)->lock) : "memory" ); \
                       } while(0)

#define spin_unlock_wait(x) do { barrier(); } \
                            while(((volatile spinlock_t *)(x))->lock == 0)

#define spin_lock(x) do {                \
	while (__ldcw (&(x)->lock) == 0) \
		while ((x)->lock == 0) ; \
        } while (0)

#else

#define SPIN_LOCK_UNLOCKED_INIT { 1, 0L, 0L }
#define SPIN_LOCK_UNLOCKED (spinlock_t) SPIN_LOCK_UNLOCKED_INIT

/* Define 6 spinlock primitives that don't depend on anything else. */

#define spin_lock_init(x)       do { (x)->lock = 1; (x)->owner_cpu = 0; (x)->owner_pc = 0; } while(0)
#define spin_is_locked(x)       ((x)->lock == 0)
void spin_lock(spinlock_t *lock);
int spin_trylock(spinlock_t *lock);
void spin_unlock(spinlock_t *lock);
#define spin_unlock_wait(x)     do { barrier(); } while(((volatile spinlock_t *)(x))->lock == 0)

#endif

#endif /* __PARISC_SPINLOCK_T_H */
