/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
** FILE      : p_hash_util.c
**
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** NAME
** 	hash.c - a set of hash routines for identifiers of length CHAR_BUF_SIZE
**
** DESCRIPTION
**	Collection of routines to create and look up a hash table.
**	The hash table should be of size 4k + 3 where k is a prime number.
**	Hashing technique is open hashing with linked lists.
**	Following is a pictorial representation of the data structures used:
**	
**	(hash table header)
**	+---------+
**	| 0:      | 
**	|---------|
**	| 1: ----------> +------------+    +-----------+
**	|---------|      |  next@ ------>  |  next@    |
**	| 2:      |      |------------|    |-----------|
**	|---------|      | identifier |    | identifier|
**	| 3:      |      |------------|    |-----------|
**	|---------|      | (anything) |    |   ..      |
**	|  :      |      |   ...      |    |           |
**	|---------|      +------------+    +-----------+
**	|  :      | 
**	|---------|
**	| k-1:    | 
**	+---------+
**
** ROUTINES
**  	hash_val(key, hash_table_size)
**  	create_hash_table(hash_table_size, record_size)
**  	find_hash_key(hash_table, search_key)
**	check_bf_insert(hash_table, insert_key)
** 	insert_hash_key(hash_table, insert_key)
**	write_hash_table(hash_table, file_name, file_ptr)
**	clear_hash_table (hash_table)
**  	dump_hash_table  (hash_table)			for debug only
**
** COMMENTS
**	The hash table used is an array of pointers to whatever record
**	structure the user defines.  ID Keys that hash to the same value, i,
**	will be placed in the linked list pointed to by hash_table[i].
**	There is one important requirement, which is that the user defined
** 	record structures must map to this structure:
**
**     typedef struct dummy {
**        struct dummy    *next;
**        char            *key;
**        ....                                (anything else you want)
**        ....
**     }  user_rec, *user_rec_ptr;
**
**  To use these routines, just make the following declarations:
**
**     extern user_rec_ptr  *create_hash_table();
**     extern user_rec_ptr  find_hash_key();
**     extern user_rec_ptr  insert_hash_key();
**     extern void          dump_hash_table();       (for debug)
**     extern int           hash_val();
**
** EXAMPLE:
**     user_rec_ptr     *hash_table;
**     user_rec_ptr     my_ptr;
**
**     hash_table = create_hash_table(a_prime_number, sizeof(user_rec));
**     my_ptr = insert_hash_key( hash_table, "IDENTIFIER  ");
**        my_ptr->my_field1 = ...;
**        my_ptr->my_field2 = ...;
**
**     if ((my_ptr = find_hash_key(hash_table, "FIND_THIS_ID")) == NULL)
**        .. key not found 
**     else
**        .. key found
**
** CHANGE LOG
**	New
**	Make hash cell records generic
**	Add clear_hash_table
**	Change key_exist() to find_hash_key()
**				Change key_insert() to insert_hash_key()
**	Add write_hash_table()
**	Add check_bf_insert()
**
*/
#include <stdio.h>
#include <string.h>

#include "std.h"
#include "util.h"

typedef	struct	hash_cell
{
    struct hash_cell	*next;
    char		*key;
} HASH_CELL,
 *HASH_CELL_PTR;

typedef HASH_CELL_PTR
 *HASH_TABLE;

/*
**			CREATE_HASH_TABLE
**
** NAME
**	create_hash_table - Creates a hash table for identifiers
**
** AUTHOR
**	Betty Lin-Chan
**
** DESCRIPTION
**	Creates a hash table as an array of pointers.
**	The argument sizeTAB (in) controls the size of the table.
**	The argument sizeREC (in) controls the size of each hash cell record
**
**  This program does something sneaky in that it actually allocates
**  an array of (sizeTAB + 2) size of pointers.
**     sizeTAB is stored in the first element of the table,  and
**     sizeREC is stored in the second element of the table.
**  But the table passed back to the caller starts at array[2],
**  and the size visible to the caller is the size requested, sizeTAB.
**
** RETURNS
**  NULL    if no more memory available
**	        otherwise, a pointer to the newly created hash table
**
*/
HASH_TABLE create_hash_table (sizeTAB,sizeREC)
int	sizeTAB;
int	sizeREC;
{
    HASH_TABLE	table;
    int	        i;

    table = (HASH_TABLE) ecalloc((sizeTAB + 2) , sizeof(HASH_CELL_PTR));

    table[0] = (HASH_CELL_PTR) sizeTAB;
    table[1] = (HASH_CELL_PTR) sizeREC;

    return (&table[2]);

} /* create_hash_table */
/*
**					HASH_VAL
**
** NAME
**	hash_val - Returns a hash cell location from a key
**
**
** DESCRIPTION
**	Generates a hash number between 1 and the argument size (in).
**	The hash function is the sum of the values of all non_blanks
**	multiplied by the first and last non-blank.
**	The argument key (in) must be a string.
**
** CAUTIONS
**	A blank keyword returns 0
**	Similarly, keywords with preceeding blanks will hash to 0
**
** RETURNS
** 	index into hash table, (0 .. hash table size-1 )
**
*/
int	hash_val (key, sizeTAB)

char   *key;
int	sizeTAB;
{
    int	first_char = *key;
    int	last_char;
    int	value = 0;

    /* Sum up values of each character in the keyword */
    while ((*key != '\0') && (*key != ' ')) {
	value += *key;
	key++;
    }

    last_char = *(--key);
    return ((value * first_char * last_char) % sizeTAB);

} /* hash_val */
/*
**			FIND_HASH_KEY
**
** NAME
**	find_hash_key - Is key in the hash table?
**
**
** DESCRIPTION
**	key  : (in) string to search for
**	table: (in) hash table to search for the key from
**
**	If key is found in the table, then pointer to the
**	hash_cell_rec containing the key is returned.
**	If the key is not found, NULL is the return value.
**
** CAUTIONS
**	A key with 1st character = blank, will return false
**	It is required that the records pointed to by hash_table pointers
**	must have this format:
**	  1st element  ---- pointer to next record
**        2nd element  ---- NULL terminated string  of length 13
**	      ...      ---- anything at all
**
** RETURNS
**	 NULL - if not found, otherwise
**       	pointer to the record containing the key is returned
**
*/
HASH_CELL_PTR	find_hash_key (table, ref_key)
HASH_TABLE  table;
char        *ref_key;
{
    HASH_CELL_PTR	ptr;
    int	        	hash_loc;
    int	        	sizeTAB;

    if ((*ref_key == ' ') || (table == NULL))
	return (NULL);

    sizeTAB  = (int) *(table - 2);
    hash_loc = hash_val(ref_key, sizeTAB);
    ptr      = table[hash_loc];

    while (ptr != NULL)			/* scan linked list */
    {
	if (strcmp(ref_key, ptr->key) == 0)
	    return (ptr);
	ptr = ptr->next;
    }

    return(NULL);
} /* find_hash_key */
/*
**			INSERT_HASH_KEY
**
** NAME
**	insert_hash_key - insert this key into the hash table
**
**
** DESCRIPTION
** 	ref_key : (in) string to hash into the hash table
** 	table   : (in) hash table created by create_hash_table()
**
** 	a new record is created and inserted at the head of the list
** 	that the given key is hashed into.
**
** RETURNS
**	 NULL - insufficient memory,
**		otherwise valid address of the newly allocated record of
**		size specified when create_hash_table(TABLEsize, RECORDsize)
**		called.
**
*/
HASH_CELL_PTR	insert_hash_key (table, ref_key)
HASH_TABLE  table;
char        *ref_key;
{
    int			hash_loc;
    HASH_CELL_PTR	new_cell;
    int			sizeTAB;
    int			sizeREC;

    if (table == NULL)
	return NULL;
    sizeTAB = (int) *(table - 2);
    sizeREC = (int) *(table - 1);

    hash_loc = hash_val(ref_key, sizeTAB);
    new_cell = (HASH_CELL_PTR) ecalloc(1, sizeREC);

    new_cell->key = ref_key;

    new_cell->next = table[hash_loc];		 /* setup pointers */
    table[hash_loc]= new_cell;

    return (new_cell);

} /* insert_hash_key */
/*
**			CLEAR_HASH_TABLE
**
** NAME
**	clear_hash_table - Clears all entries from a hash table.
**
**
** DESCRIPTION
**	Scans all table entries in the argument table(in) for hash contents,
**	unlinks and frees them.
**
*/
void	clear_hash_table (table)
HASH_TABLE	table;
{
    int		    i;
    HASH_CELL_PTR   list_ptr;
    HASH_CELL_PTR   free_ptr;
    int		    sizeTAB;

    if (table == NULL)
	return;
    sizeTAB = (int) *(table - 2);

    for (i = 0; i < sizeTAB; i++)
    {
	if (table[i] != NULL)
	{
	    list_ptr = table[i];
	    table[i] = NULL;
	    do
	    {
		free_ptr = list_ptr;
	    	list_ptr = list_ptr->next;
		free(free_ptr);
	    } while (list_ptr != NULL);
	}
    }
} /* clear_hash_table */
#ifdef DEBUG
/*
**			DUMP_HASH_TABLE
**
** NAME
**	dump_hash_table - Displays the contents of a hash table
**
**
** DESCRIPTION
**	The contents of the hash table indicated by the argument table (in)
**	is displayed.  Only those cells with data is printed, with each
**	open linked list keys on one line.
**
**
** RETURNS
**	void
*/
void	dump_hash_table (table)

HASH_TABLE	table;
{
    HASH_CELL_PTR   ptr;
    int		    i;
    int		    size;
    Boolean	    empty;

    if (table == NULL)
	return;
    size = (int) *(table - 2);

    fprintf(stderr, ld_gets(1, 1124, "Dump Hash Table, size = %d\n"), size);
    for (i = 0; i < size; i++)
    {
	empty = TRUE;

	if ((ptr = table[i]) != NULL)
	{
	    fprintf(stderr,ld_gets(1, 1125, "Element: %4d"), i);
	    empty = FALSE;
	    do
	    {
		fprintf(stderr," %s", ptr->key);
	    } while ((ptr = ptr->next) != NULL);
	}

	if (! empty)
	    fprintf(stderr,"\n");
    }

} /* dump_hash_table */

/*
**			WRITE_HASH_TABLE
**
** NAME
**	write_hash_table - write out the contents of a hash table
**
**
** DESCRIPTION
**	Write out the content of the hash table into the file pointed to.
**	calling this routine.
**	
** CAUSTIONS
**	Caller must check the integrity of the file pointer before 
**	calling this routine.
**	
** RETURNS
**	0	for no errors
**	1	for cannot write to file
*/
int write_hash_table(table, file_name, file_ptr)
HASH_TABLE   table;
char	     file_name[];
FILE	     *file_ptr;
{
    int		    i;
    HASH_CELL_PTR   ptr;
    int	      	    sizeTAB;
    int		    sizeREC;

    if (table == NULL)
	return;
    sizeTAB = (int) *(table - 2);
    sizeREC = (int) *(table - 1);

    for (i = 0; i < sizeTAB; i++)
    {
	ptr = table[i];
	while (ptr != NULL)
	{
	    if (fwrite(ptr, sizeREC, 1, file_ptr) != 1)
	    {
		printf(ld_gets(1, 
			       1126, 
			       "write_hash_table: can't write to %s?!"), 
		       file_name);
		return (1);
	    }
	    ptr = ptr->next;
	}
    }

    return (0);
}
#endif /* DEBUG */

#if 0

/* 
**
**			CHECK_BF_INESRT
**
** NAME
**	check_bf_insert - check before inserting into hash table
**
**
** DESCRIPTION
**	If the key already exist in the hash table, this routine will 
**	return a pointer to the existing record.
**	If the key does not exist in the hash table, this routine will
**	allocate a record for it, enter it into the hash table, and
**	then return a pointer to the new record.
**	
** CAUTIONS
**	The hash table passed to the routine must have been created
**	by the create_hash_table() routine in this file.
**	
** RETURNS
**	ptr	to the record containing the key
*/
HASH_CELL_PTR check_bf_insert(table, ref_key)
HASH_TABLE    table;
char	      ref_key[];
{
    int		    hash_loc;
    HASH_CELL_PTR   cell_ptr;
    int	      	    sizeTAB;
    int		    sizeREC;

    sizeTAB = (int) *(table - 2);
    sizeREC = (int) *(table - 1);

    hash_loc = hash_val(ref_key, sizeTAB);
    cell_ptr = table[hash_loc];

    while (cell_ptr != NULL)
    {
	if (strcmp(ref_key, cell_ptr->key) == 0)
	    return (cell_ptr);
	cell_ptr = cell_ptr->next;
    }

    cell_ptr = (HASH_CELL_PTR) ecalloc(1, sizeREC);
    cell_ptr->key = ref_key;
    cell_ptr->next  = table[hash_loc];
    table[hash_loc] = cell_ptr;
    return (cell_ptr);
}
#endif
