/*=====================================================================
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  

   File: elf_detector.c

   Purpose: functions to detect ELF files.

 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
=====================================================================*/

/* System and standard headers */
#include <assert.h>
#include <stddef.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <lst.h>
#include <ar.h>

#include <sys/types.h> /* off_t --pschwan@thepuffingroup.com */

#include "elf.h"
#include "elf_parisc.h"

/* headers */
#include "elf_detector.h"

#define NTH_CHAR(x, n) (((char *) (&(x)))[(n)])

#define FLIP_HWORD(dest, src, is_flip) \
   do { \
     if (is_flip) { \
       NTH_CHAR(dest, 0) = NTH_CHAR(src, 1); \
       NTH_CHAR(dest, 1) = NTH_CHAR(src, 0); \
     } else { \
       dest = src; \
     } \
   } while(0)

#define FLIP_WORD(dest, src, is_flip) \
   do { \
     if (is_flip) { \
       NTH_CHAR(dest, 0) = NTH_CHAR(src, 3); \
       NTH_CHAR(dest, 1) = NTH_CHAR(src, 2); \
       NTH_CHAR(dest, 2) = NTH_CHAR(src, 1); \
       NTH_CHAR(dest, 3) = NTH_CHAR(src, 0); \
     } else { \
       dest = src; \
     } \
   } while(0)

#define FLIP_DWORD(dest, src, is_flip) \
   do { \
     if (is_flip) { \
       NTH_CHAR(dest, 0) = NTH_CHAR(src, 7); \
       NTH_CHAR(dest, 1) = NTH_CHAR(src, 6); \
       NTH_CHAR(dest, 2) = NTH_CHAR(src, 5); \
       NTH_CHAR(dest, 3) = NTH_CHAR(src, 4); \
       NTH_CHAR(dest, 4) = NTH_CHAR(src, 3); \
       NTH_CHAR(dest, 5) = NTH_CHAR(src, 2); \
       NTH_CHAR(dest, 6) = NTH_CHAR(src, 1); \
       NTH_CHAR(dest, 7) = NTH_CHAR(src, 0); \
     } else { \
       dest = src; \
     } \
   } while(0)

static short elf_endian_values = ( (ELFDATA2MSB << 8) + ELFDATA2LSB);
static char *_elf_running_endian_p = (char *) &elf_endian_values;
#define _ELF_RUNNING_ENDIAN     (*_elf_running_endian_p)

/* the only version id expected in the version_id field of the LST header
 * in a SOM archive 
 */
#define VERSION 85082112
#define SARHDR  60 /* size of the archive header to skip */
#define SIZE    SARHDR + SLSTHDR /* total size to read */


int elfd_get_elf_object_type(int fildes)
{
   off_t seek_save;
   char elf_ident[EI_NIDENT];
   int return_type = ELFD_UNKNOWN;
   unsigned char ei_class;
   unsigned char ei_osabi;
   int do_flip;
   union {
      Elf32_Ehdr elf32_ehdr;
      Elf64_Ehdr elf64_ehdr;
   } elf_ehdr;

   assert(fildes != -1);

   seek_save = lseek(fildes, 0, SEEK_CUR);

   lseek(fildes, 0, SEEK_SET);
   if (read(fildes, elf_ident, EI_NIDENT) == EI_NIDENT) {
      if (memcmp(elf_ident, ELFMAG, SELFMAG) == 0) {
	 /* An ELF file */
	 ei_class = elf_ident[EI_CLASS];
	 do_flip = (_ELF_RUNNING_ENDIAN != elf_ident[EI_DATA]);
	 ei_osabi = elf_ident[EI_OSABI];
	 lseek(fildes, 0, SEEK_SET);
	 switch (ei_class) {
	    case ELFCLASS32:
	       /* An ELF-32 file */
	       if (read(fildes,
			&elf_ehdr,
			sizeof(Elf32_Ehdr)) == sizeof(Elf32_Ehdr)) {

		  Elf32_Half e_type;
		  Elf32_Half e_machine;
		  Elf32_Word e_flags;

		  FLIP_HWORD(e_type, elf_ehdr.elf32_ehdr.e_type, do_flip);
		  FLIP_HWORD(e_machine,
			     elf_ehdr.elf32_ehdr.e_machine,
			     do_flip);
		  FLIP_WORD(e_flags, elf_ehdr.elf32_ehdr.e_flags, do_flip);

		  if (e_type == ET_REL) {
		     if ((e_machine == EM_PARISC) &&
			 (e_flags & EF_PARISC_WIDE) &&
			 (ei_osabi == ELFOSABI_HPUX)) {
			return_type = ELFD_OBJECT_PARISC_HPUX_64BIT;
		     } else {
			return_type = ELFD_OBJECT_UNKNOWN;
		     }
		  } else if (e_type == ET_DYN) {
		     if ((e_machine == EM_PARISC) &&
			 (e_flags & EF_PARISC_WIDE) &&
			 (ei_osabi == ELFOSABI_HPUX)) {
			return_type = ELFD_SHLIB_PARISC_HPUX_64BIT;
		     } else {
			return_type = ELFD_SHLIB_UNKNOWN;
		     }
		  }
	       }
	       break;

	    case ELFCLASS64:
	    case ELFCLASS64_A:
	       /* An ELF-64 file */
	       if (read(fildes,
			&elf_ehdr,
			sizeof(Elf64_Ehdr)) == sizeof(Elf64_Ehdr)) {

		  Elf64_Half e_type;
		  Elf64_Half e_machine;
		  Elf64_Word e_flags;

		  FLIP_HWORD(e_type, elf_ehdr.elf64_ehdr.e_type, do_flip);
		  FLIP_HWORD(e_machine,
			     elf_ehdr.elf64_ehdr.e_machine,
			     do_flip);
		  FLIP_WORD(e_flags, elf_ehdr.elf64_ehdr.e_flags, do_flip);

		  if (e_type == ET_REL) {
		     if ((e_machine == EM_PARISC) &&
			 (e_flags & EF_PARISC_WIDE) &&
			 (ei_osabi == ELFOSABI_HPUX)) {
			return_type = ELFD_OBJECT_PARISC_HPUX_64BIT;
		     } else {
			return_type = ELFD_OBJECT_UNKNOWN;
		     }
		  } else if (e_type == ET_DYN) {
		     if ((e_machine == EM_PARISC) &&
			 (e_flags & EF_PARISC_WIDE) &&
			 (ei_osabi == ELFOSABI_HPUX)) {
			return_type = ELFD_SHLIB_PARISC_HPUX_64BIT;
		     } else {
			return_type = ELFD_SHLIB_UNKNOWN;
		     }
		  }
	       }
	       break;
	 }
      }
   }

   lseek(fildes, seek_save, SEEK_SET);

   return return_type;
}


int elfd_is_elf_archive(int fildes)
{
   char buf[SIZE];
   struct lst_header* lst_hdr;
   int *ptr;
   unsigned int val; /* to compute the checksum */
   int i;
   off_t seek_save;
   int return_value = 0;

   seek_save = lseek(fildes, 0, SEEK_CUR);

   assert(fildes != -1);

   /* start from the beginning of the file */
   lseek(fildes, 0, SEEK_SET);

   /* check if it's an archive (ELF or SOM) */
   if (read(fildes, buf, SARMAG) == SARMAG) {
      if (strncmp(buf, ARMAG, SARMAG) == 0) {
	 if (read(fildes, buf, SIZE) != SIZE) { 
	    /* it's an archive and it's not big enough to be a SOM library
             * therefore it must be an ELF archive 
             */
	    return_value = 1;
         } else { 
	    /* check if it's a SOM library */
	    lst_hdr = (struct lst_header*)(buf + SARHDR);
            if (lst_hdr->a_magic == LIBMAGIC && 
                lst_hdr->version_id == VERSION) {
	       /* the SOM LST header is present -- still could be a 
		  freak coincidence -- check the checksum to make sure */
	       ptr = (int *)lst_hdr;
               val = 0;
               for (i = 0; i < (SLSTHDR/sizeof(int))-1; i++) {
		  val ^= *ptr++;
               }
               return (val == lst_hdr->checksum) ? 0: 1;
            } else { /* a_magic and version_id do not tally -- must be ELF */
	       return_value = 1;
            }
         }
      }
   }

   lseek(fildes, seek_save, SEEK_SET);

   return return_value;  /* not an elf archive */
}
