/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
 *
***************************************************************************
**
** File:         annotate.c
**
** Description:  Support for WW annotations
**
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***************************************************************************
*/
#ifdef WW_ANNOTATIONS
#include <stdlib.h>
#include <stdio.h>
#include <syms.h>
#ifndef DEBUG
#define NDEBUG
#endif
#include <assert.h>

#include "std.h"
#include "ldlimits.h"
#include "spacehdr.h"
#include "scnhdr.h"
#include "annotate.h"
#include "spaces.h"
#include "subspaces.h"
#include "driver.h"
#include "util.h"
#include "symbols.h"
#include "linker.h"

/******************************************************************************
 *                 OVERVIEW OF GLOBAL ANNOTATIONS DATA                        *
 ******************************************************************************

 annot_bits_used:    This keeps track of how many bits are currently used in
                     the cur_annot_word.
 cur_annot_word:     This holds the next word of data for the annotations bit
		     stream as we build it.
 annotation_buf_ptr: This is a pointer to the array of annotation records that
		     are buffered up per region before we convert them to
		     the output bit stream.
 annotation_buf_max_entries: This is the size of the array pointed to by
		     annotation_buf_ptr.
 annotation_buf_index: This is the current index of the next free entry in
		     the array pointed to by annotation_buf_ptr.
 last_flushed_flags: This is the value of the last set of flags flushed to
		     the output bit stream.
 annot_subsp_ptr:    This is a pointer to the current position in the output
		     subspace where out_bits will put more data.
 do_ww_annotations:  This is controls the generation of annotations.  By
		     default the linker always generates annotations.
 annotations_killed_at_startup: If annotations were killed at startup, either
		     by -FA, or an incompatible option (-A, -r or fru), this
		     is set true, and we traverse the subspaces in input order
		     for better performance.
 annot_subsp_index:  Index of the subspace that contains the output bit stream.
 annotatins_size_bits: The number of bits needed to hold the annotations
		     output bit stream for this link
******************************************************************************/

static unsigned int annot_bits_used=0, cur_annot_word=0;
static struct annotation_rec *annotation_buf_ptr=NULL;
static int annotation_buf_max_entries=0, annotation_buf_index=0;
static int last_flushed_flags=0;

static unsigned int *annot_subsp_ptr;

#if 0 /*  No longer generate PA annotations. */
int do_ww_annotations=TRUE;	/* Default: Always generate WW annotations! */
int annotations_killed_at_startup=FALSE;
#endif /* 0 */
/* 11.01 we do not generate WW annotations
 * anymore and the default is changed to FALSE. */
int do_ww_annotations=FALSE;	
int annotations_killed_at_startup=TRUE;
int annot_subsp_index = BAD_SUBSP;
int annotations_size_bits=0;

/* Forward declarations */
static void out_pad_opcodes(unsigned int pad_len, int mode);
static void out_header_with_cont(unsigned int len, unsigned int flags,
				 int subsp, int mode);

void init_annotations(void)
{
  int i;

  /* First we build a subspace to hold our annotations data */
  annot_subsp_index = build_subspace("$CI$","$ANS$",ANS_SORTKEY,0x1f,
				      CODE_QUADRANT,CI_SORTKEY,4,0,0);

  /* Now, reset some fields to be accurate! */
  space_array[Subspace_Sp_Number(annot_subsp_index)].is_loadable = FALSE;
  Subsp_Dict(annot_subsp_index).is_loadable = FALSE;
} /* end init_annotations() */

void init_annotation_subsp(void)
{
  /* Need to set initialization_length, subspace_length and subsp_data
     when we know this after long_reach_stubs_added()! */
  /* We also need to put out the header at this time! */
  unsigned int annotations_size_bytes;
  char *data;
  int old_index;

  /* Compute the size of the annotations in bytes */
  annotations_size_bytes = annotations_size_bits/8;
  /* Add one byte if there is any remainder */
  if (annotations_size_bits & 0x7) {
    annotations_size_bytes++;
  }

  Subsp_Dict(annot_subsp_index).initialization_length=annotations_size_bytes;
  Subsp_Dict(annot_subsp_index).subspace_length=annotations_size_bytes;
  /*
  ** Malloc an extra word of data so I can safely write out the contents
  ** of cur_annot_word at the end of flush_current_annotations().  See
  ** the comment there for more detail.
  */
  data=emalloc(annotations_size_bytes + sizeof(cur_annot_word));
  Subsp_Misc(annot_subsp_index).subsp_data=data;
  annot_subsp_ptr=(unsigned int *)data;
  *annot_subsp_ptr=ANNOTATIONS_HEADER;
  annot_subsp_ptr++;
#ifdef DEBUG
  if (verbose & V_WW_ANNOTATIONS)
    printf("annotations_size_bits=%d\n",annotations_size_bits);
#endif /* DEBUG */
  last_flushed_flags=0;
  annotation_buf_index=0;
  old_index=Subsp_Misc(0).r_n_x;
  out_annotation(START_OF_TEXT,BAD_SUBSP,Subspace_Virtual_Offset(old_index),
		 0,0,0);
} /* end init_annotation_subsp() */

static void out_bits(unsigned int num_bits, unsigned int value)
{
  unsigned int mask,fragment;

  /* Left justify the bits in the word.  This makes things easier */
  /* If num_bits == 32, this is a NOP */
  value=value<<(32 - num_bits);

  if (num_bits > 32 - annot_bits_used) {
    /* Take the top 32-annot_bits_used bits of value */
    /* Note! I want to shift the constant only 32-annot_bits_used-1,
       or 31-annot_bits_used to make the mask */
    mask=((int)0x80000000)>>(31-annot_bits_used);
    /* Time to sub-clone the bits
       First we digest the bits */
    fragment = value & mask;
    /* Now we shift the bits over */
    fragment >>= annot_bits_used;
    /* Now we ligate them into place */
    cur_annot_word |= fragment;

    /* Adjust the input parameters */
    value <<= 32-annot_bits_used;
    num_bits -= 32-annot_bits_used;

    /* Save this word to the output subspace and adjust pointers */
    *annot_subsp_ptr = cur_annot_word;
    annot_subsp_ptr++;
    cur_annot_word=0;
    annot_bits_used =0;
  }

  /* The remaining bits are guaranteed to fit in cur_annot_word */
  /* Shift the bits over */
  value >>= annot_bits_used;
  /* Ligate them into place */
  cur_annot_word |= value;

  /* Adjust pointers */
  annot_bits_used += num_bits;

  if (annot_bits_used == 32) {
    /* We have filled this word, so dump it out */
    *annot_subsp_ptr=cur_annot_word;
    annot_subsp_ptr++;
    cur_annot_word=0;
    annot_bits_used=0;
  }

} /* end out_bits() */

void out_annotation(int type, int subsp, int addr, unsigned int info,
	       unsigned int value, unsigned int new_flags)
{
  struct annotation_rec *new_ptr;

#define ANNOTATION_SEG_SIZE 200

  if (! do_ww_annotations) {
     return;
  }

  if (annotation_buf_index >= annotation_buf_max_entries) {
    /* We have overflowed our buffer.  Time to realloc()! */
    annotation_buf_ptr=(struct annotation_rec *) erealloc((char *)annotation_buf_ptr,
	     (annotation_buf_max_entries+ANNOTATION_SEG_SIZE) * 
	         sizeof(struct annotation_rec));
    annotation_buf_max_entries += ANNOTATION_SEG_SIZE;
  }
  new_ptr=annotation_buf_ptr + annotation_buf_index;
  new_ptr->type = (char)type;
  /* Do that 0 = 32 hack thing */
  if (type==COPY_BITS && info == 0) {
    info=32;
  }
  new_ptr->info = (unsigned char)info;
  new_ptr->new_flags = (unsigned char)new_flags;
  new_ptr->subsp = subsp;
  new_ptr->addr = addr;
  new_ptr->value = value;
  new_ptr->reserved1 = 0;

  annotation_buf_index++;

} /* end out_annotation() */

void reset_annotation_buffer(void)
{
  int old_index;

  if (! do_ww_annotations) {
     return;
  }

#ifdef DEBUG
  if (verbose & V_WW_ANNOTATIONS)
    printf("Reset annotations buffer\nAdded 32 bits for annotations header\n");
#endif /* DEBUG */
  annotations_size_bits=32;
  last_flushed_flags=0;
  annotation_buf_index=0;
  old_index=Subsp_Misc(0).r_n_x;
  out_annotation(START_OF_TEXT,BAD_SUBSP,Subspace_Virtual_Offset(old_index),
		 0,0,0);
} /* end reset_annotation_buffer() */

void flush_current_annotations(int new_subsp_index, unsigned int new_addr,
			       int mode)
{
  int i;
  unsigned int subsp_end, pad_len, len, temp;
  int new_flags_op;
#define NO_FLAGS_CHANGED -1
  Boolean need_end_op, final_flush=FALSE;

  /* If there are no saved up annotations,
     or we aren't doing WW annotations, just return */
  if (annotation_buf_index <= 0 || !do_ww_annotations) {
    annotation_buf_index=0;
    return;
  }

  need_end_op=FALSE;

  /* Special case.  If this is the last set of annotations to flush,
     then fool the following code into working... */
  if (new_subsp_index == BAD_SUBSP && new_addr == 0) {
    if (annotation_buf_ptr->type == START_OF_TEXT) {
      /* There is nothing to flush, just return */
      return;
    } else {
      final_flush=TRUE;
      new_subsp_index = annotation_buf_ptr->subsp; 
      new_addr = Subspace_Virtual_Offset(new_subsp_index) +
	         Subspace_Length(new_subsp_index);
    }
  }

  switch (annotation_buf_ptr->type) {
    case START_OF_TEXT:
      if (annotation_buf_ptr->addr != new_addr) {
	/* We need to make a 0 len region with padding */
	temp=0;
	/* Get padding len in words */
	pad_len=(new_addr - annotation_buf_ptr->addr)/4;
	/* If we get a negative pad, we are in trouble */
	assert(pad_len >= 0);
	if (pad_len > 1) {
	  temp|=ANNOT_MORE;
	  need_end_op=TRUE;
	}
	/* Set the lsb of the pad len if necessary */
	if (pad_len & 0x1) {
	  temp|=ANNOT_PAD_LSB;
        }

        if (mode == FLUSH_MODE) {
          out_bits(11,temp);
        } else {
	  annotations_size_bits+=11;
        }
#ifdef DEBUG
        if (verbose & V_WW_ANNOTATIONS)
          printf("Added 11 bits for START_OF_TEXT header opcode\n");
#endif /* DEBUG */

	/* Get pad len in double words */
	pad_len>>=1;
        if (pad_len) {
	  out_pad_opcodes(pad_len,mode);
        }
      }
      break;

    case STUBS_HEADER:
      pad_len=(new_addr - (Subspace_Virtual_Offset(annotation_buf_ptr->subsp) +
	      Subsp_Stubs_Size(annotation_buf_ptr->subsp))) / 4;
      /* If we get a negative pad, we are in trouble */
      assert(pad_len >= 0);

      /* Get the stubs size in words */
      len=Subsp_Stubs_Size(annotation_buf_ptr->subsp)/4;
      temp=ANNOT_IS_STUB;
      if (len>0xff || pad_len>1) {
	temp |= ANNOT_MORE;
	need_end_op=TRUE;
      }
      if (pad_len & 0x1) {
	temp |= ANNOT_PAD_LSB;
      }
      out_header_with_cont(len,temp,annotation_buf_ptr->subsp,mode);

      /* Convert pad_len to double words */
      pad_len >>= 1;
      if (pad_len) {
	out_pad_opcodes(pad_len,mode);
      }
      break;

    case ENTRY_HEADER:
      /* Add the size of the header */

      /* compute new flags opcode if necessary */
      new_flags_op = NO_FLAGS_CHANGED;
      if (annotation_buf_ptr->new_flags != last_flushed_flags) {
	new_flags_op=(ANNOT_GFLAGS<<4)| annotation_buf_ptr->new_flags;
	last_flushed_flags = annotation_buf_ptr->new_flags;
      }

      /* Compute pad length and region length */
      if (annotation_buf_ptr->subsp != new_subsp_index) {
	subsp_end = Subspace_Virtual_Offset(annotation_buf_ptr->subsp) +
		    Subspace_Length(annotation_buf_ptr->subsp);
	/* Get pad len in words */
	pad_len=(new_addr - subsp_end)/4;
        /* If we get a negative pad, we are in trouble */
        assert(pad_len >= 0);

	/* We will compute the length of the region based on the end
	   of our subspace when we change subspaces */
	len = (subsp_end - annotation_buf_ptr->addr)/4;
      } else {
	/* We can't have a padding in the same subspace */
	pad_len=0;
	len = (new_addr - annotation_buf_ptr->addr)/4;
      }

      /* Compute our header, and set any flags necessary */
      temp=0;
      if (pad_len>1 || len>0xff || (new_flags_op != NO_FLAGS_CHANGED) ||
	  annotation_buf_index>1) {
        temp |= ANNOT_MORE;
	need_end_op=TRUE;
      }
      if (pad_len & 0x1) {
	temp |= ANNOT_PAD_LSB;
      }
      out_header_with_cont(len,temp,annotation_buf_ptr->subsp,mode);

      /* Convert pad_len to length in double words */
      pad_len >>= 1;
      if (pad_len) {
	out_pad_opcodes(pad_len,mode);
      }

      if (new_flags_op != NO_FLAGS_CHANGED) {
	if (mode == FLUSH_MODE) {
	  out_bits(8,new_flags_op);
	} else {
	  annotations_size_bits+=8;
	}
#ifdef DEBUG
	if (verbose & V_WW_ANNOTATIONS)
	  printf("Added 8 bits for new flags=0x%x\n",
		  annotation_buf_ptr->new_flags);
#endif /* DEBUG */
      }
      break;

    default:
#ifdef DEBUG 
      fprintf(stderr,"Annotation error: First annotation to flush wasn't a STUBS_HEADER\n or ENTRY_HEADER or START_OF_TEXT header!\n");
#endif /* DEBUG */
      stop_annotations();
      return;
  }

  if (annotation_buf_index>1 && annotation_buf_ptr->type==STUBS_HEADER) {
#ifdef DEBUG
    fprintf(stderr,"STUBS_HEADER has extra annotation opcodes!\n");
#endif /* DEBUG */
    stop_annotations();
    return;
  }

  /* For each additional annotation */
  for (i=1; i<annotation_buf_index; i++) {
    /* Set this if we need to count an end opcode */
    switch ((annotation_buf_ptr+i)->type) {
      case COPY_BITS:
	if (mode == FLUSH_MODE) {
	  out_bits((annotation_buf_ptr+i)->info, (annotation_buf_ptr+i)->value);
	} else {
	  annotations_size_bits+=(annotation_buf_ptr+i)->info;
	}
#ifdef DEBUG
	if (verbose & V_WW_ANNOTATIONS)
	  printf("Added %d bits for COPY_BITS annotation\n",
		  (annotation_buf_ptr+i)->info);
#endif /* DEBUG */
	break;

      case SYM_ADDR_LOOKUP:
	/* get the opcode */
	temp = (annotation_buf_ptr+i)->info << 4;
	/* get the sub-opcode */
	temp |= (annotation_buf_ptr+i)->value >> 28;
	if (mode == FLUSH_MODE) {
	  out_bits(8,temp);
	} else {
	  annotations_size_bits+=8;
	}
	/* Get the symbol index */
	temp = (annotation_buf_ptr+i)->value & 0xfffffff;
	/* Bias it  */
	temp += Subsp_Misc((annotation_buf_ptr+i)->subsp).symbol_index_bias;
	if ((temp >= sym_dict_size) || (Sym_Type(temp) == ST_SYM_EXT) ||
	    (Sym_Type(temp) == ST_ARG_EXT) || (Sym_Subsp(temp) == BAD_SUBSP)) {
	  /* If we are given a bogus symbol index, just stop making
	     annotations */
	  stop_annotations();
	  return;
	}
	/* Pass it an index, get a virtual address */
	temp = symbol_value(temp);
	if (mode == FLUSH_MODE) {
	  out_bits(32,temp);
	} else {
	  annotations_size_bits+=32;
	}
#ifdef DEBUG
	if (verbose & V_WW_ANNOTATIONS)
	  printf("Added 40 bits for SYM_ADDR_LOOKUP annotation\n");
#endif /* DEBUG */
	break;

      default:
#ifdef DEBUG
        fprintf(stderr,"Annotation error: Unknown type of annotation\n");
#endif /* DEBUG */
	stop_annotations();
	return;
    }
  }

  if (need_end_op) {
    if (mode == FLUSH_MODE) {
      out_bits(4,ANNOT_END);
    } else {
      annotations_size_bits+=4;
    }
#ifdef DEBUG
    if (verbose & V_WW_ANNOTATIONS)
      printf("Added 4 bits for END opcode\n");
#endif /* DEBUG */
  }

  annotation_buf_index=0;

  if (final_flush && mode == FLUSH_MODE) {
    /* We probably don't need to write the entire contents of cur_annot_word,
    ** because not all the bits may be used. However, I can safely save it
    ** because I malloc'ed the needed size + sizeof(cur_annot_word) in
    ** init_annotations_subsp(), so I am guaranteed not to overwrite any
    ** data past the end of the buffer.  This extra space is only for the
    ** malloc'ed region, and does not affect the subspace size written out
    ** to the file.
    */
    *annot_subsp_ptr=cur_annot_word;

    /* Make sure we didn't overshoot the buffer for holding these things. */
    assert ((int) annot_subsp_ptr <=
            (int) Subsp_Misc(annot_subsp_index).subsp_data + 
	    (int) Subsp_Dict(annot_subsp_index).subspace_length );
  }
} /* end flush_current_annotatins() */

void stop_annotations(void)
{
  /* If this is a debug linker, just die with an assert */
  assert(0);

  annotation_buf_index=0;
  do_ww_annotations=FALSE;
  if (annot_subsp_index != BAD_SUBSP) {
    if (Subsp_Misc(annot_subsp_index).subsp_data) {
      efree(annot_subsp_ptr);
    }
    delete_subsp(annot_subsp_index);
    annot_subsp_index=BAD_SUBSP;
  }
} /* end stop_annotations() */


static void out_header_with_cont(unsigned int len, unsigned int flags,
				 int subsp, int mode)
{
  if (len > 0xff) {
    out_header_with_cont(len>>4, flags, subsp, mode);
    /* Now we output the extension opcode needed */
    if (mode == FLUSH_MODE) {
      out_bits(8,(ANNOT_CONT<<4)|(len & 0xf));
    } else {
      annotations_size_bits+=8;
    }
#ifdef DEBUG
    if (verbose & V_WW_ANNOTATIONS)
      printf("Added 8 bits for continuation for len=0x%x\n", len&0xf);
#endif /* DEBUG */
  } else {
    /* We have a number small enough to fit in the pad opcode */
    if (mode == FLUSH_MODE) {
      out_bits(11,(len<<3)|flags);
    } else {
      annotations_size_bits+=11;
    }
#ifdef DEBUG
    if (verbose & V_WW_ANNOTATIONS)
      printf("Added 11 bits for header for subsp %d, len=0x%x\n",
	      subsp, len&0xff);
#endif /* DEBUG */
  }
} /* end out_header_with_cont() */

static void out_pad_opcodes(unsigned int pad_len, int mode)
{
  if (pad_len > 0xff) {
    out_pad_opcodes(pad_len>>4, mode);
    /* Now we output the extension opcode needed */
    if (mode == FLUSH_MODE) {
      out_bits(8,(ANNOT_CONT<<4)|(pad_len & 0xf));
    } else {
      annotations_size_bits+=8;
    }
#ifdef DEBUG
    if (verbose & V_WW_ANNOTATIONS)
      printf("Added 8 bits for continuation for pad len=0x%x\n", pad_len&0xf);
#endif /* DEBUG */
  } else {
    /* We have a number small enough to fit in the pad opcode */
    if (mode == FLUSH_MODE) {
      out_bits(12,(ANNOT_PAD<<8)|pad_len);
    } else {
      annotations_size_bits+=12;
    }
#ifdef DEBUG
    if (verbose & V_WW_ANNOTATIONS)
      printf("Added 12 bits for padding opcode, len=0x%x\n",pad_len&0xff);
#endif /* DEBUG */
  }
} /* end out_pad_opcodes() */
#endif /* WW_ANNOTATIONS */

