#include <lbz2.h>

/*=========================================================================*/
/* init.c -- */

#if 0 /* PHI */
#include "lib_bzip.h"
#include "lib_bzip_private.h"
#endif

#define MAX(a,b) ((a)>(b)?(a):(b))

void *bzip2_work_area;

size_t
bzip_init(void *mem)
{
   bzip2_work_area = mem;
   return MAX( sizeof(Lib_Bzip_Encode_Storage_Ty ),
	       sizeof(Lib_Bzip_Decode_Storage_Ty ));
   /* sizeof(encoding space) == 150516,
      sizeof(decoding space) == 151552
      (though this may have changed slightly by the time you read
      this comment). */
}

/*=========================================================================*/
/* lib_bzip_huffman.c -- */

/*-------------------------------------------------------------*/
/*-- Huffman coding low-level stuff                          --*/
/*--                                      lib_bzip_huffman.c --*/
/*-------------------------------------------------------------*/

/*--
  This file is part of lib_bzip, a block-sorting compression 
  library designed to compress 32kbyte blocks, for on-the-fly
  disk compression/decompression.

  Version 0.02, 4-Jan-1998

  Copyright (C) 1996, 1997, 1998 by Julian Seward.
     Guildford, Surrey, UK
     email: jseward@acm.org

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--*/

#if 0 /* PHI */
#include "lib_bzip_private.h"
#endif


/*---------------------------------------------------*/
/*--- Huffman coding low-level stuff              ---*/
/*---------------------------------------------------*/

#define WEIGHTOF(zz0)  ((zz0) & 0xffffff00U)
#define DEPTHOF(zz1)   ((zz1) & 0x000000ffU)
#define MYMAX(zz2,zz3) ((zz2) > (zz3) ? (zz2) : (zz3))

#define ADDWEIGHTS(zw1,zw2)                           \
   (WEIGHTOF(zw1)+WEIGHTOF(zw2)) |                    \
   (1 + MYMAX(DEPTHOF(zw1),DEPTHOF(zw2)))

#define UPHEAP(z)                                     \
do {                                                  \
   Int32 zz, tmp;                                     \
   zz = z; tmp = heap[zz];                            \
   while (weight[tmp] < weight[heap[zz >> 1]]) {      \
      heap[zz] = heap[zz >> 1];                       \
      zz >>= 1;                                       \
   }                                                  \
   heap[zz] = tmp;                                    \
} while(0)

#define DOWNHEAP(z)                                   \
do {                                                  \
   Int32 zz, yy, tmp;                                 \
   zz = z; tmp = heap[zz];                            \
   while (True) {                                     \
      yy = zz << 1;                                   \
      if (yy > nHeap) break;                          \
      if (yy < nHeap &&                               \
          weight[heap[yy+1]] < weight[heap[yy]])      \
         yy++;                                        \
      if (weight[tmp] < weight[heap[yy]]) break;      \
      heap[zz] = heap[yy];                            \
      zz = yy;                                        \
   }                                                  \
   heap[zz] = tmp;                                    \
} while(0)


/*---------------------------------------------*/
void hbMakeCodeLengths ( UChar* len, 
                         Int32* freq,
                         Int32  alphaSize,
                         Int32  maxLen,
                         // temporaries
                         Int32* heap,
                         Int32* weight,
                         Int32* parent)
{
   /*--
      Nodes and heap entries run from 1.  Entry 0
      for both the heap and nodes is a sentinel.
   --*/
   Int32 nNodes, nHeap, n1, n2, i, j, k;
   Bool  tooLong;

   /*--
      Now passed in from caller, so as to keep stack
      frame small.
   Int32 heap   [ MAX_ALPHA_SIZE + 2 ];
   Int32 weight [ MAX_ALPHA_SIZE * 2 ];
   Int32 parent [ MAX_ALPHA_SIZE * 2 ]; 
   --*/

   for (i = 0; i < alphaSize; i++)
      weight[i+1] = (freq[i] == 0 ? 1 : freq[i]) << 8;

   while (True) {

      nNodes = alphaSize;
      nHeap = 0;

      heap[0] = 0;
      weight[0] = 0;
      parent[0] = -2;

      for (i = 1; i <= alphaSize; i++) {
         parent[i] = -1;
         nHeap++;
         heap[nHeap] = i;
         UPHEAP(nHeap);
      }
   
      while (nHeap > 1) {
         n1 = heap[1]; heap[1] = heap[nHeap]; nHeap--; DOWNHEAP(1);
         n2 = heap[1]; heap[1] = heap[nHeap]; nHeap--; DOWNHEAP(1);
         nNodes++;
         parent[n1] = parent[n2] = nNodes;
         weight[nNodes] = ADDWEIGHTS(weight[n1], weight[n2]);
         parent[nNodes] = -1;
         nHeap++;
         heap[nHeap] = nNodes;
         UPHEAP(nHeap);
      }

      tooLong = False;
      for (i = 1; i <= alphaSize; i++) {
         j = 0;
         k = i;
         while (parent[k] >= 0) { k = parent[k]; j++; }
         len[i-1] = j;
         if (j > maxLen) tooLong = True;
      }
      
      if (! tooLong) break;

      for (i = 1; i < alphaSize; i++) {
         j = weight[i] >> 8;
         j = 1 + (j / 2);
         weight[i] = j << 8;
      }
   }
}


/*---------------------------------------------*/
void hbAssignCodes ( UInt32 *code,
                     UChar *length,
                     unsigned minLen,
                     unsigned maxLen,
                     Int32 alphaSize )
{
   Int32 i;
   UInt32 vec;
   unsigned n;

   vec = 0;
   for (n = minLen; n <= maxLen; n++) {
      for (i = 0; i < alphaSize; i++)
         if (length[i] == n) { code[i] = vec; vec++; };
      vec <<= 1;
   }
}


/*---------------------------------------------*/
void hbCreateDecodeTables ( Int32 *limit,
                            Int32 *base,
                            Int32 *perm,
                            UChar *length,
                            Int32 minLen,
                            Int32 maxLen,
                            Int32 alphaSize )
{
   Int32 pp, i, j, vec;

   pp = 0;
   for (i = minLen; i <= maxLen; i++)
      for (j = 0; j < alphaSize; j++)
         if (length[j] == i) { perm[pp] = j; pp++; };

   for (i = 0; i < MAX_CODE_LEN; i++) base[i] = 0;
   for (i = 0; i < alphaSize; i++) base[length[i]+1]++;

   for (i = 1; i < MAX_CODE_LEN; i++) base[i] += base[i-1];

   for (i = 0; i < MAX_CODE_LEN; i++) limit[i] = 0;
   vec = 0;

   for (i = minLen; i <= maxLen; i++) {
      vec += (base[i+1] - base[i]);
      limit[i] = vec-1;
      vec <<= 1;
   }
   for (i = minLen + 1; i <= maxLen; i++)
      base[i] = ((limit[i-1] + 1) << 1) - base[i];
}


/*-------------------------------------------------------------*/
/*-- end                                  lib_bzip_huffman.c --*/
/*-------------------------------------------------------------*/


/*=========================================================================*/
/* lib_bzip_e.c -- */
/*-------------------------------------------------------------*/
/*-- bzip-like compression of 32k blocks        lib_bzip_e.c --*/
/*-------------------------------------------------------------*/

/*--
  This file is part of lib_bzip, a block-sorting compression 
  library designed to compress 32kbyte blocks, for on-the-fly
  disk compression/decompression.

  Version 0.02, 4-Jan-1998

  Copyright (C) 1996, 1997, 1998 by Julian Seward.
     Guildford, Surrey, UK
     email: jseward@acm.org

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--*/

#if 0 /* PHI */
#include "lib_bzip_private.h"
#include "lib_bzip.h"
#endif


/*---------------------------------------------*/
/*--
   Compressed block format:

      UChar 'B'
      UChar 'Z'
      UChar 2         -- version 0.02 of this lib
      
      (a single bit indicating whether or not this
       block was randomised)

      UInt16 origPtr  -- rotation posn of original

      (table indicating which chars are in use)

      (Huffman code length tables, one for each
       of N_GROUPS)

      (selector table)

      (codes for the MTF values proper)

--*/


static Lib_Bzip_Encode_Storage_Ty *est;

# define e_verbosity est->verb

/*---------------------------------------------*/
/*-- 
   Bit stream writing stuff (boring but necessary).
--*/

static void 
bsInit_e ( void )
{
   est->outctr = 0;
   est->bsLive = 0;
   est->bsBuff = 0;
}


#define bs_MAKE_SPACE                         \
do {                                          \
   while (est->bsLive >= 8) {                 \
      est->outbuff[est->outctr++]             \
         = (UChar)(est->bsBuff >> 24);        \
      est->bsBuff <<= 8;                      \
      est->bsLive -= 8;                       \
   }                                          \
} while(0)


static void bsW ( Int32 n, UInt32 v )
{
   bs_MAKE_SPACE;
   est->bsBuff |= (v << (32 - est->bsLive - n));
   est->bsLive += n;
}


static void 
bsFlush_e ( void )
{
   while (est->bsLive > 0) {
      est->outbuff[est->outctr++]
         = (UChar)(est->bsBuff >> 24);
      est->bsBuff <<= 8;
      est->bsLive -= 8;
   }
}


/*---------------------------------------------*/
static inline
Bool 
fullGt ( Int32 i1, Int32 i2 )
{
   UInt16 s1, s2;
   Int32 k;

   if (i1 == i2) return False;

   s1 = est->block[i1]; s2 = est->block[i2];
   if (s1 != s2) return (s1 > s2);
   i1 += 2; i2 += 2;

   s1 = est->block[i1]; s2 = est->block[i2];
   if (s1 != s2) return (s1 > s2);
   i1 += 2; i2 += 2;

   s1 = est->block[i1]; s2 = est->block[i2];
   if (s1 != s2) return (s1 > s2);
   i1 += 2; i2 += 2;

   k = est->nblock + 16;

   do {

      s1 = est->block[i1]; s2 = est->block[i2];
      if (s1 != s2) return (s1 > s2);
      i1 += 2; i2 += 2;

      s1 = est->block[i1]; s2 = est->block[i2];
      if (s1 != s2) return (s1 > s2);
      i1 += 2; i2 += 2;

      s1 = est->block[i1]; s2 = est->block[i2];
      if (s1 != s2) return (s1 > s2);
      i1 += 2; i2 += 2;

      if (i1 >= (Int32) est->nblock)
         i1 -= est->nblock;
      if (i2 >= (Int32) est->nblock)
         i2 -= est->nblock;

      k -= 6;
      est->workDone ++;
   }
      while (k >= 0);

   return False;
}


/*---------------------------------------------*/
/*  This qsort is derived from Weiss' book 
    "Data Structures and Algorithm Analysis in C",
    Section 7.7.
*/

#define ISORT_BELOW 10

#define SWAP(za,zb)                                               \
do { Int32 zl = (za); Int32 zr = (zb);                            \
     UInt16 zt = est->zptr[zl]; est->zptr[zl] = est->zptr[zr];    \
     est->zptr[zr] = zt;                                          \
   } while(0)

static void 
sortBucket ( Int32 left, Int32 right )
{
   UInt16 v;
   Int32 pivot;
   Int32 i, j;
   Int32 wuC;

   UInt16 stackL[20];
   UInt16 stackR[20];

   Int32 sp = 0;
   Int32 wuL = left;
   Int32 wuR = right;

   if (left >= right) return;

   while (True) {

      /*-- 
         At the beginning of this loop, wuL and wuR hold the
         bounds of the next work-unit.  First, though, check
         that we are not out of CPU time.
      --*/

      if (wuR - wuL > ISORT_BELOW) {

         /*--
            a large Work Unit; partition-exchange 
         --*/
         wuC = (wuL + wuR) >> 1;
         if (fullGt ( 1+(Int32)est->zptr[wuL], 1+(Int32)est->zptr[wuC] ))
            SWAP ( wuL, wuC );
         if (fullGt ( 1+(Int32)est->zptr[wuL], 1+(Int32)est->zptr[wuR] ))
            SWAP ( wuL, wuR );
         if (fullGt ( 1+(Int32)est->zptr[wuC], 1+(Int32)est->zptr[wuR] ))
            SWAP ( wuC, wuR );

         SWAP ( wuC, wuR-1 );
         pivot = 1 + (Int32)est->zptr[wuR-1];

         i = wuL;
         j = wuR - 1;
         for (;;) {
            do i++; while (fullGt ( pivot, 1+(Int32)est->zptr[i] ));
            do j--; while (fullGt ( 1+(Int32)est->zptr[j], pivot ));
            if (i < j) SWAP ( i, j ); else break;
            if (!est->randomised && est->workDone > est->workLimit) return;
         }
         SWAP ( i, wuR-1 );

         if ((i - wuL) > (wuR - i)) {
            stackL[sp] = wuL; stackR[sp] = i-1; sp++; wuL = i+1;
         } else {
            stackL[sp] = i+1; stackR[sp] = wuR; sp++; wuR = i-1;
         }

      } else {

         /*--
            a small Work-Unit; insertion-sort it
         --*/                
         for (i = wuL + 1; i <= wuR; i++) {
            v = est->zptr[i];
            j = i;
            while ( fullGt ( 1+(Int32)est->zptr[j-1], 1+(Int32)v )) {
               est->zptr[j] = est->zptr[j-1];
               j = j - 1;
               if (j <= wuL) break;
            }
            est->zptr[j] = v;
            if (!est->randomised && est->workDone > est->workLimit) return;
         }
         if (sp == 0) return;
         sp--; wuL = stackL[sp]; wuR = stackR[sp];

      } /* if this is a small work-unit */
   }
}

#undef RC
#undef SWAP
#undef ISORT_BELOW




/*---------------------------------------------*/
#define BIGFREQ(zz) (est->ftab[(zz)+1] - est->ftab[(zz)])

#define FREQ(zz)    (est->ftab[(zz)*3 +3] - est->ftab[(zz)*3 +0])
#define FREQ_LT(zz) (est->ftab[(zz)*3 +1] - est->ftab[(zz)*3 +0])
#define FREQ_EQ(zz) (est->ftab[(zz)*3 +2] - est->ftab[(zz)*3 +1])
#define FREQ_GT(zz) (est->ftab[(zz)*3 +3] - est->ftab[(zz)*3 +2])

#define TIMES3(zz) ((zz)+(zz)+(zz))

static void 
sortBlock ( void )
{
   Int32 j, k, totalOrdered, bucket;
   unsigned i;
   UInt16 s, sLo, sHi;
   Int32 ftabNo;

   // stripe the block, and set up overshoot area

   if (est->nblock == 1) {
      est->block[0] &= 0xff00;
      est->block[0] = (est->block[0] >> 8) | est->block[0];
   } else {
      for (i = 0; i < est->nblock - 1; i++) {
         est->block[i] &= 0xff00;
         est->block[i] |= (est->block[i+1] >> 8);
      }
      est->block[est->nblock-1] &= 0xff00;
      est->block[est->nblock-1] |= (est->block[0] >> 8);
   }

   for (i = 0; i < N_OVERSHOOT; i++)
      est->block[i + est->nblock] = est->block[i % est->nblock];


   // set up freq table

   for (i = 0; i < 769; i++) est->ftab[i] = 0;

   for (i = 0; i < est->nblock; i++) {
      s = est->block[i];
      sHi = s >> 8;
      sLo = s & 0xFF;
      if (sHi > sLo) ftabNo = TIMES3(sHi) + 0; else
      if (sHi < sLo) ftabNo = TIMES3(sHi) + 2; else
                     ftabNo = TIMES3(sHi) + 1;
      est->ftab[ftabNo]++;
   }

   for (i = 1; i < 769; i++) est->ftab[i] += est->ftab[i-1];

   for (i = 0; i < est->nblock; i++) {
      s = est->block[i];
      sHi = s >> 8;
      sLo = s & 0xFF;
      if (sHi > sLo) ftabNo = TIMES3(sHi) + 0; else
      if (sHi < sLo) ftabNo = TIMES3(sHi) + 2; else
                     ftabNo = TIMES3(sHi) + 1;
      j = est->ftab[ftabNo] - 1;
      est->ftab[ftabNo] = j;
      est->zptr[j] = i;
   }

   for (i = 0; i < 256; i++) est->inUse[i] = FREQ(i) > 0;

   for (i = 0; i < 256; i++) est->rorder[i] = i;
   
   {
      Int32 vv;
      Int32 h = 1;
      do h = 3 * h + 1; while (h <= 256);
      do {
         h = h / 3;
         for (i = h; i <= 255; i++) {
            vv = est->rorder[i];
            j = i;
            while ( FREQ(est->rorder[j-h]) > FREQ(vv) ) {
               est->rorder[j] = est->rorder[j-h];
               j = j - h;
               if (j < (Int32) h) break;
            }
            est->rorder[j] = vv;
         }
      } while (h != 1);
   }

   totalOrdered = 0;
   for (i = 0; i < 256; i++) {

      bucket = est->rorder[i];

      if (FREQ_LT(bucket) > 1) {
         sortBucket ( est->ftab[TIMES3(bucket) +0], 
                      est->ftab[TIMES3(bucket) +1]-1 );
         totalOrdered += FREQ_LT(bucket);
         if (!est->randomised && est->workDone > est->workLimit) return;
      }

      if (FREQ_GT(bucket) > 1) {
         sortBucket ( est->ftab[TIMES3(bucket) +2], 
                      est->ftab[TIMES3(bucket) +3]-1 );
         totalOrdered += FREQ_LT(bucket);
         if (!est->randomised && est->workDone > est->workLimit) return;
      }

      if (FREQ_EQ(bucket) > 1) {
         Int32 put0, get0, put1, get1;
         UChar c1;
         Int32 sbn = TIMES3(bucket) + 1;
         Int32 lo = est->ftab[sbn];
         Int32 hi = est->ftab[sbn+1] - 1;
         UChar ssc = (UChar)bucket;

         put0 = lo;
         get0 = est->ftab[TIMES3(bucket)];
         put1 = hi;
         get1 = est->ftab[TIMES3(bucket)+3] - 1;
         while (get0 < put0) {
            j = est->zptr[get0]-1; if (j < 0) j += est->nblock;
            c1 = (UChar)(est->block[j] >> 8);
            if (c1 == ssc) { est->zptr[put0] = j; put0++; };
            get0++;
         }
         while (get1 > put1) {
            j = est->zptr[get1]-1; if (j < 0) j += est->nblock;
            c1 = (UChar)(est->block[j] >> 8);
            if (c1 == ssc) { est->zptr[put1] = j; put1--; };
            get1--;
         }
         totalOrdered += FREQ_EQ(bucket);
      }

      if (i < 255 && FREQ(bucket) < 256) {
         Int32 loLim = est->ftab[TIMES3(bucket)];
         Int32 hiLim = est->ftab[TIMES3(bucket) + 3];
         for (j = 0, k = loLim;   k < hiLim;   j++, k++) {
            UInt16 a2upd = est->zptr[k];
            est->block[a2upd] = (est->block[a2upd] & 0xff00) | ((UInt16)j);
            if (a2upd < N_OVERSHOOT)
               est->block[a2upd + est->nblock] 
                  = (est->block[a2upd + est->nblock] & 0xff00) | ((UInt16)j);
         }
      }

   }
}

#undef FREQ
#undef FREQ_LT
#undef FREQ_EQ
#undef FREQ_GT
#undef TIMES3


/*---------------------------------------------*/
static void makeMaps_e ( void )
{
   unsigned i;

   est->nInUse = 0;
   for (i = 0; i < 256; i++)
      if (est->inUse[i]) {
         est->unseqToSeq[i] = est->nInUse;
         est->nInUse++;
      }
}



/*---------------------------------------------*/
static void 
generateMTFValues ( void )
{
   UChar   yy[256];
   unsigned i;
   Int32   j;
   UChar   tmp;
   UChar   tmp2;
   Int32   zPend;
   Int32   wr;
   Int32   EOB;

   makeMaps_e ();
   EOB = est->nInUse + RUNBASE - 1;

   for (i = 0; (Int32) i <= EOB; i++) est->mtfFreq[i] = 0;

   wr = 0;
   zPend = 0;
   for (i = 0; (Int32) i < est->nInUse; i++)
      yy[i] = (UChar) i;

   for (i = 0; i < est->nblock; i++) {
      UChar ll_i;

      j = est->zptr[i] - 1; if (j < 0) j = est->nblock-1;      
      ll_i = est->unseqToSeq[est->block[j] >> 8];

      j = 0;
      tmp = yy[j];
      while ( ll_i != tmp ) {
         j++;
         tmp2 = tmp;
         tmp = yy[j];
         yy[j] = tmp2;
      };
      yy[0] = tmp;

      if (j == 0) {
         zPend++;
      } else {
         if (zPend > 0) {
            zPend--;
            while (True) {
               switch (zPend % RUNBASE) {
                  case 0: est->zptr[wr] = RUNA; wr++; est->mtfFreq[RUNA]++; break;
                  case 1: est->zptr[wr] = RUNB; wr++; est->mtfFreq[RUNB]++; break;
               };
               if (zPend < RUNBASE) break;
               zPend = (zPend - RUNBASE) / RUNBASE;
            };
            zPend = 0;
         }
         est->zptr[wr] = j+RUNBASE-1; wr++; est->mtfFreq[j+RUNBASE-1]++;
      }
   }

   if (zPend > 0) {
      zPend--;
      while (True) {
         switch (zPend % RUNBASE) {
            case 0:  est->zptr[wr] = RUNA; wr++; est->mtfFreq[RUNA]++; break;
            case 1:  est->zptr[wr] = RUNB; wr++; est->mtfFreq[RUNB]++; break;
         };
         if (zPend < RUNBASE) break;
         zPend = (zPend - RUNBASE) / RUNBASE;
      };
   }

   est->zptr[wr] = EOB; wr++; est->mtfFreq[EOB]++;

   est->nMTF = wr;
}


/*---------------------------------------------*/
#define LESSER_ICOST  0
#define GREATER_ICOST 15

static Bool
sendMTFValues ( void )
{
   Int32 v, t, i, j, gs, ge, totc, bt, bc, iter;
   Int32 alphaSize, selCtr;
   UInt31 nSelectors;
   Int32 nBytes;

   /*--
   UChar  len [N_GROUPS][MAX_ALPHA_SIZE];
   is a global since the decoder also needs it.

   UInt32 code[N_GROUPS][MAX_ALPHA_SIZE];
   Int32  rfreq[N_GROUPS][MAX_ALPHA_SIZE];
   are also globals only used in this proc.
   Made global to keep stack frame size small.
   --*/

   UInt16 cost[N_GROUPS];
   Int32  fave[N_GROUPS];

   alphaSize = est->nInUse + RUNBASE;
   for (t = 0; t < N_GROUPS; t++)
      for (v = 0; v < alphaSize; v++)
         est->len[t][v] = GREATER_ICOST;

   /*--- Generate an initial set of coding tables ---*/
   { 
      Int32 nPart, remF, tFreq, aFreq;

      nPart = N_GROUPS;
      remF  = est->nMTF;
      gs = 0;
      while (nPart > 0) {
         tFreq = remF / nPart;
         ge = gs-1;
         aFreq = 0;
         while (aFreq < tFreq && ge < alphaSize-1) {
            ge++;
            aFreq += est->mtfFreq[ge];
         }

         if (ge > gs 
             && nPart != N_GROUPS && nPart != 1 
             && ((N_GROUPS-nPart) % 2 == 1)) {
            aFreq -= est->mtfFreq[ge];
            ge--;
         }
         for (v = 0; v < alphaSize; v++)
            if (v >= gs && v <= ge) 
               est->len[nPart-1][v] = LESSER_ICOST; else
               est->len[nPart-1][v] = GREATER_ICOST;
 
         nPart--;
         gs = ge+1;
         remF -= aFreq;
      }
   }

   /*--- 
      Iterate up to N_ITERS times to improve the tables.
   ---*/
   for (iter = 0; iter < N_ITERS; iter++) {

      for (t = 0; t < N_GROUPS; t++) fave[t] = 0;

      for (t = 0; t < N_GROUPS; t++)
         for (v = 0; v < alphaSize; v++)
            est->rfreq[t][v] = 0;

      nSelectors = 0;
      totc = 0;
      gs = 0;
      while (True) {

         /*--- Set group start & end marks. --*/
         if (gs >= est->nMTF) break;
         ge = gs + G_SIZE - 1; 
         if (ge >= est->nMTF) ge = est->nMTF-1;

         /*-- 
            Calculate the cost of this group as coded
            by each of the coding tables.
         --*/
         for (t = 0; t < N_GROUPS; t++) cost[t] = 0;

         {
            register UInt16 cost0, cost1;
            cost0 = cost1 = 0;
            for (i = gs; i <= ge; i++) { 
               UInt16 icv = est->zptr[i];
               cost0 += est->len[0][icv];
               cost1 += est->len[1][icv];
            }
            cost[0] = cost0; cost[1] = cost1;
         }
 
         /*-- 
            Find the coding table which is best for this group,
            and record its identity in the selector table.
         --*/
         bc = 999999999; bt = -1;
         for (t = 0; t < N_GROUPS; t++)
            if (cost[t] < bc) { bc = cost[t]; bt = t; };
         totc += bc;
         fave[bt]++;
         est->selector[nSelectors] = bt;
         nSelectors++;

         /*-- 
            Increment the symbol frequencies for the selected table.
          --*/
         for (i = gs; i <= ge; i++)
            est->rfreq[bt][ est->zptr[i] ]++;

         gs = ge+1;
      }

      /*--
        Recompute the tables based on the accumulated frequencies.
      --*/
      for (t = 0; t < N_GROUPS; t++)
         hbMakeCodeLengths ( 
            &(est->len[t][0]), &(est->rfreq[t][0]), alphaSize, 20,
            &(est->heap_tmp[0]), &(est->weight_tmp[0]), &(est->parent_tmp[0])
         );
   }

   /*--- Assign actual codes for the tables. --*/
   for (t = 0; t < N_GROUPS; t++) {
      unsigned minLen = 32;
      unsigned maxLen = 0;

      for (i = 0; i < alphaSize; i++) {
         if (est->len[t][i] > maxLen)
	    maxLen = est->len[t][i];
         if (est->len[t][i] < minLen)
	    minLen = est->len[t][i];
      }
      hbAssignCodes ( &(est->code[t][0]), &(est->len[t][0]), 
                      minLen, maxLen, alphaSize );
   }

   /*--- Transmit the mapping table. ---*/
   { 
      Bool inUse16[16];

      for (i = 0; i < 16; i++) {
          inUse16[i] = 0;
          for (j = 0; j < 16; j++)
             if (est->inUse[i * 16 + j])
		inUse16[i] = 1;
      }
     
      nBytes = est->outctr;
      for (i = 0; i < 16; i++)
	 bsW(1, inUse16[i]);

      for (i = 0; i < 16; i++) {
         if (inUse16[i]) {
            for (j = 0; j < 16; j++) {
               if (est->inUse[i * 16 + j]) { 
			bsW(1,1);
		} else { 
			bsW(1,0);
		}
	    }
         }
     }

   }

   /*--- Now the selectors. ---*/
   nBytes = est->outctr;
   bsW ( 10, nSelectors );
   for (i = 0; i < (Int32) nSelectors; i++)
      if (est->selector[i] == 0) bsW(1,0); else bsW(1,1);

   /*--- Now the coding tables. ---*/
   nBytes = est->outctr;

   for (t = 0; t < N_GROUPS; t++) {
      unsigned curr = est->len[t][0];
      bsW ( 5, curr );
      for (i = 0; i < alphaSize; i++) {
         while (curr < est->len[t][i]) { bsW(2,2); curr++; /* 2 -> binary 10 */ };
         while (curr > est->len[t][i]) { bsW(2,3); curr--; /* 3 -> binary 11 */ };
         bsW ( 1, 0 );
      }
   }

   /*--
      Compute the total size of output, and check whether 
      it will fit into the caller's output buffer, and, 
      if not, give up now.
   --*/
   {
      Int32 totalBits = 8 * est->outctr;

      selCtr = 0;
      gs = 0;
      while (True) {
         UChar* ltab;
         if (gs >= est->nMTF) break;
         ge = gs + G_SIZE - 1; 
         if (ge >= est->nMTF) ge = est->nMTF-1;
         ltab = &(est->len[est->selector[selCtr]][0]);
         for (i = gs; i <= ge; i++)
            totalBits += ltab[est->zptr[i]];
         gs = ge+1;
         selCtr++;
      }
      /*-- allow for worst-case flushing of bit buffer --*/
      totalBits += 32;
      /*-- convert to bytes --*/
      totalBits = (totalBits + 7) / 8;
      est->predictedSize = totalBits;
      if (totalBits > est->totalSizeAllowed) return False;
   }


   /*--- And finally, the block data proper ---*/
   nBytes = est->outctr;
   selCtr = 0;
   gs = 0;
   while (True) {
      UChar*  ltab;
      UInt32* ctab;
      if (gs >= est->nMTF) break;
      ge = gs + G_SIZE - 1; 
      if (ge >= est->nMTF) ge = est->nMTF-1;
      ltab = &(est->len [est->selector[selCtr]][0]);
      ctab = &(est->code[est->selector[selCtr]][0]);
      for (i = gs; i <= ge; i++) {
         UInt16 zptr_i = est->zptr[i];
         bsW ( ltab[zptr_i], ctab[zptr_i] );
      }
      gs = ge+1;
      selCtr++;
   }

   return True;
}


/*---------------------------------------------*/
Int16 rNums[64] = {    
   619, 720, 127, 481, 931, 816, 813, 233, 566, 247, 
   985, 724, 205, 454, 863, 491, 741, 242, 949, 214, 
   733, 859, 335, 708, 621, 574, 73, 654, 730, 472, 
   419, 436, 278, 496, 867, 210, 399, 680, 480, 51, 
   878, 465, 811, 169, 869, 675, 611, 697, 867, 561, 
   862, 687, 507, 283, 482, 129, 807, 591, 733, 623, 
   150, 238, 59, 379
};


static void 
randomiseBlock ( void )
{
   unsigned i;
   RAND_DECLS;

   for (i = 0; i < est->nblock; i++) {
      RAND_UPD_MASK;
      est->block[i] ^= RAND_MASK << 8;
   }
}


/*---------------------------------------------*/
size_t 
bzip_compressBlock ( unsigned char *in_buf, 
		     unsigned char *out_buf,
		     size_t num_in_buf,
		     size_t num_out_buf,
		     int param /* unused */ )
{
   size_t i;
   Bool fitsOk;

   if (num_in_buf == 0)
      return 0; /* Some functions (e.g. sortBlock() don't like nblock to equal 0. */

   est = (Lib_Bzip_Encode_Storage_Ty *) bzip2_work_area;

   est->totalSizeAllowed = num_out_buf;
   est->verb             = 0;
   est->nblock           = num_in_buf;

   // make est->block a copy of in_buf <<\'d 8

   for (i = 0; i < num_in_buf; i++)
      est->block[i] = ((UInt16)in_buf[i]) << 8;

   // try to sort it
   est->randomised = False;
   est->workLimit  = 50 * est->nblock;
   est->workDone   = 0;
   sortBlock ();
   // if sorting failed because of too much work, 
   // randomise and try again (this can\'t fail).

   if (est->workDone > est->workLimit) {
      randomiseBlock ();
      est->randomised = True;
      est->workLimit = 0;
      est->workDone = 0;
      sortBlock ();
   }

   // find the posn of the original string

   i = 0;
   while (i < est->nblock && est->zptr[i] != 0)
      i++;

   est->outbuff = out_buf;
   bsInit_e ();
   bsW ( 8, (UChar)'B' ); bsW ( 8, (UChar)'Z' );
   bsW ( 8, 2 );

   if (est->randomised) bsW(1,1); else bsW(1,0);

   bsW ( 16, i );

   generateMTFValues ();

   fitsOk = sendMTFValues ();
   if (!fitsOk) return 0;

#ifndef __KERNEL__
   if (est->outctr > est->predictedSize) {
      fprintf ( stderr, "panic: predicted size is too small\n" );
      fprintf ( stderr, "       predicted %d, actual %d\n", 
                       est->predictedSize, est->outctr );
      exit(1);
   }
#endif

   bsFlush_e ();

   return est->outctr;
}


/*-------------------------------------------------------------*/
/*-- end                                        lib_bzip_e.c --*/
/*-------------------------------------------------------------*/

/*======================================================================*/
/* lib_bzip_d.c -- */

/*-------------------------------------------------------------*/
/*-- bzip-like decompression of 32KB blocks     lib_bzip_d.c --*/
/*-------------------------------------------------------------*/

/*--
  This file is part of lib_bzip, a block-sorting compression 
  library designed to compress 32kbyte blocks, for on-the-fly
  disk compression/decompression.

  Version 0.02, 4-Jan-1998

  Copyright (C) 1996, 1997, 1998 by Julian Seward.
     Guildford, Surrey, UK
     email: jseward@acm.org

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--*/

#if 0 /* PHI */
#include "lib_bzip_private.h"
#include "lib_bzip.h"
#endif

#include <string.h>

static Lib_Bzip_Decode_Storage_Ty *dst;




/*---------------------------------------------*/
/*--
   Bit stream reading stuff.
--*/

static void 
bsInit_d ( void )
{
   dst->inctr = 0;
   dst->bsLive = 0;
   dst->bsBuff = 0;
}


#define bsNEEDR(nz)                                   \
do {	                                              \
   while (dst->bsLive < nz) {                          \
      Int32 zzi = (Int32)(dst->inbuff[dst->inctr++]);   \
      dst->bsBuff = (dst->bsBuff << 8) | (zzi & 0xffL); \
      dst->bsLive += 8;                                \
   }                                                  \
} while(0)


static inline
UInt32 
bsR ( Int32 n )
{
   UInt32 v;
   bsNEEDR ( n );
   v = (dst->bsBuff >> (dst->bsLive-n)) & ((1 << n)-1);
   dst->bsLive -= n;
   return v;
}

/*---------------------------------------------*/
static void 
makeMaps_d ( void )
{
   Int32 i;
   dst->nInUse = 0;
   for (i = 0; i < 256; i++)
      if (dst->inUse[i]) {
         dst->seqToUnseq[dst->nInUse] = i;
         dst->nInUse++;
      }
}


/*---------------------------------------------*/
static void 
recvDecodingTables ( void )
{
   Int32  i, j, t, nSelectors, alphaSize;
   Int32  minLen, maxLen;
   Bool inUse16[16];

   dst->origPtr = bsR ( 16 );

   /*--- Receive the mapping table ---*/
   for (i = 0; i < 16; i++)
      if (bsR(1) == 1) 
         inUse16[i] = True; else 
         inUse16[i] = False;

   for (i = 0; i < 256; i++) dst->inUse[i] = False;

   for (i = 0; i < 16; i++)
      if (inUse16[i])
         for (j = 0; j < 16; j++)
            if (bsR(1) == 1) dst->inUse[i * 16 + j] = True;

   makeMaps_d ();
   alphaSize = dst->nInUse + RUNBASE;

   /*--- Now the selectors ---*/
   nSelectors = bsR ( 10 );
   for (i = 0; i < nSelectors; i++)
      dst->selector[i] = bsR(1)==0 ? 0 : 1;

   /*--- Now the coding tables ---*/
   for (t = 0; t < N_GROUPS; t++) {
      Int32 curr = bsR ( 5 );
      for (i = 0; i < alphaSize; i++) {
         while (bsR(1) == 1) {
            if (bsR(1) == 0) curr++; else curr--;
         }
         dst->len[t][i] = curr;
      }
   }

   /*--- Create the Huffman decoding tables ---*/
   for (t = 0; t < N_GROUPS; t++) {
      minLen = 32;
      maxLen = 0;
      for (i = 0; i < alphaSize; i++) {
         if (dst->len[t][i] > maxLen) maxLen = dst->len[t][i];
         if (dst->len[t][i] < minLen) minLen = dst->len[t][i];
      }
      hbCreateDecodeTables ( 
         &(dst->limit[t][0]), &(dst->base[t][0]),
         &(dst->perm[t][0]), &(dst->len[t][0]),
         minLen, maxLen, alphaSize
      );
      dst->minLens[t] = minLen;
   }
}


/*---------------------------------------------*/
#define GET_MTF_VAL(lval)                 \
do {                                      \
   Int32 zn, zvec, zj;                    \
   if (groupPos == 0) {                   \
      groupNo++;                          \
      groupPos = G_SIZE;                  \
      gSel = dst->selector[groupNo];       \
      gMinlen = dst->minLens[gSel];        \
      gLimit = &(dst->limit[gSel][0]);     \
      gPerm = &(dst->perm[gSel][0]);       \
      gBase = &(dst->base[gSel][0]);       \
   }                                      \
   groupPos--;                            \
   zn = gMinlen; zvec = bsR ( zn );       \
   while (zvec > gLimit[zn]) {            \
      zn++; zj = bsR ( 1 );               \
      zvec = (zvec << 1) | zj;            \
   };                                     \
   lval = gPerm[zvec - gBase[zn]];        \
} while(0)


/*---------------------------------------------*/
static void 
decodeBlock ( UInt32* cc, 
              UChar*  mtfa, 
              Int32*  mtfbase )
{
   Int32  nextSym;
   Int32  EOB, groupNo, groupPos, gSel;
   Int32  gMinlen = 0;
   Int32* gLimit = NULL;
   Int32* gBase = NULL;
   Int32* gPerm = NULL;

   Int32 lastR, op;

   /*--
      Now passed in from caller, so as to keep stack
      frame small.
   UInt32 cc[256];
   UChar  mtfa   [MTFA_SIZE];
   Int32  mtfbase[256 / MTFL_SIZE];
   --*/

   recvDecodingTables ();

   EOB      = dst->nInUse + RUNBASE - 1;
   groupNo  = -1;
   groupPos = 0;

   memset(cc, 0, 256 * sizeof(*cc));

   //-- MTF init
   {
      unsigned i, j, k;

      k = MTFA_SIZE-1;
      for (i = 256 / MTFL_SIZE; i-- > 0;) {
         for (j = MTFL_SIZE; j-- > 0;) {
            mtfa[k] = (UChar)(i * MTFL_SIZE + j);
            k--;
         }
         mtfbase[i] = k + 1;
      }
   }
   //-- end MTF init

   lastR = -1;

   GET_MTF_VAL(nextSym);

   while (True) {

      if (nextSym < RUNBASE) {
         UChar ch;
         Int32 sum = -1;
         Int32 N = 1;
         do {
            if (nextSym == RUNA) sum = sum + (0+1) * N;
            else if (nextSym == RUNB) sum = sum + (1+1) * N;
            N = N * RUNBASE;
            GET_MTF_VAL(nextSym);
         }
            while (nextSym < RUNBASE);

         sum++;
         ch = dst->seqToUnseq[ mtfa[mtfbase[0]] ];

         while (sum > 0) {
            lastR++;
            dst->tvec[lastR] = (ch << 24) | cc[ch];
            cc[ch]++;
            sum--;
         };

         continue;

      } else if (nextSym == EOB) {

         break;

      } else {

         UChar tmp, ch;
         lastR++;

         //-- tmp = MTF ( nextSym-(RUNBASE-1) );
         {
            Int32 i, j, k, n, p, lno, off;
            n = (Int32)(nextSym - (RUNBASE-1) );

            if (n < MTFL_SIZE) { // avoid general-case expense
               p = mtfbase[0];
               tmp = mtfa[p+n];
               while (n > 3) {
                  Int32 z = p+n;
                  mtfa[(z)  ] = mtfa[(z)-1];
                  mtfa[(z)-1] = mtfa[(z)-2];
                  mtfa[(z)-2] = mtfa[(z)-3];
                  mtfa[(z)-3] = mtfa[(z)-4];
                  n -= 4;
               }
               while (n > 0) { mtfa[(p+n)] = mtfa[(p+n)-1]; n--; };
               mtfa[p] = tmp;
            } else { // general case
               lno = n / MTFL_SIZE;
               off = n % MTFL_SIZE;
               p = mtfbase[lno] + off;
               tmp = mtfa[p];
               while (p > mtfbase[lno]) { mtfa[p] = mtfa[p-1]; p--; };
               mtfbase[lno]++;
               while (lno > 0) {
                  mtfbase[lno]--;
                  mtfa[mtfbase[lno]] = mtfa[mtfbase[lno-1] + MTFL_SIZE - 1];
                  lno--;
               }
               mtfbase[0]--;
               mtfa[mtfbase[0]] = tmp;
               if (mtfbase[0] == 0) {
                  k = MTFA_SIZE-1;
                  for (i = 256 / MTFL_SIZE-1; i >= 0; i--) {
                     for (j = MTFL_SIZE-1; j >= 0; j--) {
                        mtfa[k] = mtfa[mtfbase[i] + j];
                        k--;
                     }
                     mtfbase[i] = k + 1;
                  }
	       }
            }
         }
         //-- end tmp = MTF ( nextSym-(RUNBASE-1) );

         ch = dst->seqToUnseq[tmp];
         dst->tvec[lastR] = (ch << 24) | cc[ch]; 
         cc[ch]++;

         GET_MTF_VAL(nextSym);
         continue;
      }
   }

   {
      UInt32 k;
      unsigned i;
      Int32 sum;

      k = dst->origPtr;
      sum = 0;
      for (i = 0; i < 256; i++) {
	 sum += cc[i];
	 cc[i] = sum - cc[i];
      }

      op = lastR+1;
      dst->nblock = op;
      while (op > 0) {
	 UInt32 u;
	 UChar v;
	 u = dst->tvec[k]; 
	 v = u >> 24; 
	 k = (u & 0x00ffffff) + cc[v];
	 op--; dst->outbuff[op] = v;
      }
   }
}


/*---------------------------------------------*/
static void 
unrandomiseBlock ( void )
{
   Int32 i;
   RAND_DECLS;

   for (i = 0; i < dst->nblock; i++) {
      RAND_UPD_MASK;
      dst->outbuff[i] ^= RAND_MASK;
   }
}

/*---------------------------------------------*/
size_t 
bzip_uncompressBlock ( unsigned char *in_buf, 
		       unsigned char *out_buf,
		       size_t num_in_buf, /* unused */
		       size_t num_out_buf, /* unused */
		       int param ) /* unused */
{
   UChar ch;

   dst = (Lib_Bzip_Decode_Storage_Ty *) bzip2_work_area;
   dst->inbuff = in_buf;
   dst->outbuff = out_buf;

   bsInit_d ();

   ch = bsR ( 8 ); if (ch != 'B') return 0;
   ch = bsR ( 8 ); if (ch != 'Z') return 0;
   ch = bsR ( 8 ); if (ch != 2) return 0;

   dst->randomised = False;
   if (bsR(1) == 1) dst->randomised = True;

   decodeBlock ( &(dst->cc_tmp[0]), &(dst->mtfa_tmp[0]), &(dst->mtfbase_tmp[0]) );

   if (dst->randomised) unrandomiseBlock();

   return dst->nblock;
}


/*-------------------------------------------------------------*/
/*-- end                                        lib_bzip_d.c --*/
/*-------------------------------------------------------------*/
