/* This small lib contain all needed function to retrieve a dump     */
/* saved in any partition (it can even be a file). It is possible    */
/* to save it as chunck (ie: core file) or as an image (a bitmap     */
/* that describe pages, and then pages) that can be used by P4 tools */

#include <klib.h>
#include <asm/kl_arch.h>
#include <asm/kl_mem.h>
#include <linux/dump.h>
#include <asm/dump.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <cksum.h>
#include <crashtool.h>
#include <lbz2.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/ioctl.h>

#define MAX_LONG_LONG   0xffffffffffffffff
#define MAX_STR 128
static char *Version = "2";
char	Dev[4096];
#define RootDev "/dev"
dump_page_t dp;
dump_header_t dh;
dump_header_asm_t dha;
int offset;
int lbz2_init_flag=0;
Lib_Bzip_Decode_Storage_Ty map_lbz2_decode;

/*
 * check_type_dev() -- run the stat() operation without having all the
 *                     inline definitions conflicted by the kernel.  Make
 *                     sure <path> is a block device type.
 */
int
check_type_dev(char *path)
{
        struct stat sbuf;
	int ret=0;

	if ( verbose_set == 1 ) {
		savecrash_gentitle("Checking dump device",lentitle);
		fflush(stdout);
	}
        if (stat(path, &sbuf) < 0) {
                return (1);
        }
	if ( S_ISBLK(sbuf.st_mode) ) {
		if ( verbose_set == 1 ) {
			printf("OK\n");
		}
		ret=0;
	} else {
		printf("ERROR: can't open dump device.\n");
		ret=1;
	}

        return (ret);
} 

/*
 * Name: kl_dump_retrieve()
 * Func: Gather a dump from a given device and save it into the dump
 *       directory.  The dump directory points to the possibility of
 *       a "bounds" file, which would be read, and be used to save the
 *       index number of the dump.  Returns 0 on failure, 1 if the
 *       core dump is short or if there was a problem finishing the
 *       read/write of the dump, or 2 on success.
 * 
 * 13/03/2003: added image format: The file is build with a bitmap
 *             for each page. The secret of this format is in the INDEX
 *             file. The size of the bitmap is nbp/8 rounded to the first
 *             32 and aligned on 4K page.And then add the data. 
 *             The first version take the total number of pages, and create
 *             fixe size bitmap, so the resulting files will have different size.
 *             Now take care that this bitmap it big indian like, ie:
 *             page 0 1 2 3 4 5 6 7  8  9 10 11 12 13 14 15
 *             Bit  7 6 5 4 3 2 1 0 15 14 13 12 11 10 9  8
 */

int kl_dump_retrieve(char *dumpdev, char *dumpdir, int local_debug)
{
	char buffer[DUMP_PAGE_SZ];
	char pagebuf[DUMP_PAGE_SZ];
	char msgbuf[256],tmpbuf[256];
        int bounds, devf, outf, indf, addr_set=0, chunk_nb=0, chunk_new=0, dumpdev_index=0 ;
        char *tfile, *dfile, *tmpstr, *ptrstr, *ofile, *uncompr_page, *ptr_page;
	time_t curtime,dumptime;
	uint32_t *dumptime_s;
	uint64_t start_addr,chunk_size_max,chunk_size, page_size,start_page=0,end_page=0;
	unsigned long sum_dump=0,nb_physpages=0,toffset,res=0;
	int	i,ret=0,bool=0,loop=0,index;
	uint8_t	*bitmap;
	uint8_t bytes_in_bitmap=0;
	uint32_t bitmap_max=0,bitmap32=0,bitmap_size=0,bitmap_chunk=0,bitmap_last=0,bitmap_alloc=0,bitmap_alloc_last=0;


	if ( old_fashion_flag == 1 ) {
		Version="1";
	} else {
		Version="2";
	}
        /* check the types */
        if (check_type_dev(dumpdev) != 0) {
                fprintf(KL_ERRORFP,
                        "Error: kl_check_type() on dumpdev failed!\n");
                return (-1);
        }

        /* check the types */
        if (check_type_dir(dumpdir) != 0) {
                fprintf(KL_ERRORFP,
                        "Error: kl_check_type() on dumpdir failed!\n");
                return (-1);
        }

        /* allocate the buffer for file data */
        tfile = (char *)malloc(KL_PAGE_SIZE);
        ofile = (char *)malloc(KL_PAGE_SIZE);
        dfile = (char *)malloc(KL_PAGE_SIZE);
	uncompr_page = (char *)malloc(KL_PAGE_SIZE);

        /* try to read the bounds file */
        bounds = kl_bounds_retrieve(dumpdir);

	/* try to validate the dump device */
	if ( (devf=kl_find_magic(dumpdev,buffer)) < 0) {
		return devf;
	}

	if ( verbose_set == 1 ) {
		printf("Ok\n");
		printf("Configured devices:\n");
		fflush(stdout);
	}
	for (index=0; index < MAX_DUMPDEV; index++) {
		if ( dh.dh_dump_device[index] != 0 ) {
			sprintf(msgbuf," -dump dev[%d]: 0x%x ",index,dh.dh_dump_device[index]);
			if ( kl_read_dir(RootDev,dh.dh_dump_device[index]) == 1 ) {
				if (strcmp(dumpdev,Dev)){
					sprintf (tmpbuf,"%s %s\n",msgbuf,Dev);
				} else {
					sprintf (tmpbuf,"%s %s*\n",msgbuf,Dev);
					dumpdev_index=index;
				}
			} else {
				sprintf (tmpbuf,"no device file found.\n");
			}
			printf("%s",tmpbuf);
		}
        }
	if ( verbose_set == 1 ) {
		savecrash_gentitle("Start retrieving modules",lentitle);
		fflush(stdout);
	}

	/* create the directory */
	sprintf(tfile, "%s/crash.%d", dumpdir, bounds);
	if ((outf = mkdir(tfile,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)) < 0 ) {
		fprintf(KL_ERRORFP,
				"Error: mkdir() of dump directory  \"%s\" failed!\n",
				tfile);
		close(devf);
		return (-1);
	}

	sprintf(tfile, "%s/crash.%d/INDEX", dumpdir, bounds);
	if ((indf = open(tfile, O_CREAT|O_RDWR|O_TRUNC,(S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
		fprintf(KL_ERRORFP, "Error: open() of INDEX file \"%s\" failed!\n",tfile);
		close(devf);
		return (-1);
	}

	/* Now create the INDEX */
	sprintf(tfile,"%-9s %s\n","comment","savecrash crash dump INDEX file");
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-9s %s\n","version",Version);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-9s %s\n","OS",dh.dh_utsname_sysname);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-9s %s\n","hostname",dh.dh_utsname_nodename);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-9s %s\n","modelname",dh.dh_utsname_machine);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-9s %s\n","panic",dh.dh_panic_string);
	write(indf, tfile, strlen (tfile));
	if ( dh.dh_time.tv_sec == 0 ) {
		/* it probably means that it is a dump created by a 64bit kernel */
		/* but we have some trouble with bits alignement */
		dumptime_s=(uint32_t *)&dh.dh_time;
		dumptime_s++;
		dumptime_s++;
	}
	memcpy(&dumptime,dumptime_s, sizeof(dumptime));
	sprintf(tfile,"%-9s %s","dumptime",ctime(&(dumptime)));
	write(indf, tfile, strlen (tfile));
	curtime = time((time_t *)0);
	sprintf(tfile,"%-9s %s","savetime",ctime(&curtime));
	write(indf, tfile, strlen (tfile));
	switch(dh.dh_dump_compress) {
		case DUMP_COMPRESS_NONE:
			sprintf(tfile,"%-9s %s\n","compress","DUMP_COMPRESS_NONE");
			break;
		case DUMP_COMPRESS_RLE:
			sprintf(tfile,"%-9s %s\n","compress","DUMP_COMPRESS_RLE");
			break;
		case DUMP_COMPRESS_GZIP:
			sprintf(tfile,"%-9s %s\n","compress","DUMP_COMPRESS_GZIP");
			break;
		case DUMP_COMPRESS_LBZ2:
			sprintf(tfile,"%-9s %s\n","compress","DUMP_COMPRESS_LBZ2");
			break;
		default:
			sprintf(tfile,"%-9s %s\n","compression","UNKNOWN");
	}
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-9s %s%s\n","release",dh.dh_utsname_release,dh.dh_utsname_version);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-9s %lld\n\n","memsize",(long long)dh.dh_memory_size*dh.dh_page_size);
	write(indf, tfile, strlen (tfile));

        /* now make sure we have more than just a header -- if not, leave */
        
	if (dh.dh_dump_level == DUMP_LEVEL_HEADER) {
                close(devf);
                kl_bounds_update(dumpdir, bounds + 1);
                return (-1);
        }

	/* now retrieve module list */
	/* first module is the kernel */
	sprintf(tfile,"%s",dh.dh_boot_kernel);
	sum_dump=cksum(tfile);
	if ( sum_dump != 0 ) {
		tmpstr=strrchr(dh.dh_boot_kernel,'/');
		tmpstr++;
		sprintf(ofile,"%s/crash.%d/%s",dumpdir, bounds,tmpstr);
		if ( (res=kl_cp_file(tfile,ofile)) > 0 ) {
			sprintf(tfile,"%-9s %s %s %lu %lu\n","module",dh.dh_boot_kernel,tmpstr,res,sum_dump);
			write(indf, tfile, strlen (tfile));
		} else {
			sprintf(tfile,"%-9s %s\n","module","failed to copy kernel file");
			write(indf, tfile, strlen (tfile));
		}
	} else {
		sprintf(tfile,"%-9s %s\n","module","failed to open kernel file !");
		write(indf, tfile, strlen (tfile));
	}
	strcpy(uncompr_page,dh.dh_dump_modules);
	tmpstr=uncompr_page;
	for(i=1;i<dh.dh_dump_nbr_modules;i++) {
		ptrstr=strchr(tmpstr,';');
		if ( ptrstr != NULL ) {
			ptrstr[0]='\0';
			sprintf(tfile,"%s",tmpstr);
		}
		if ( kl_find_module(tfile) != 0 ) {
			sprintf(dfile,"%-9s %s%s\n","module","can't find module ",tmpstr);
		} else {
			sum_dump=cksum(tfile);
			if ( sum_dump != 0 ) {
				tmpstr=strrchr(tfile,'/');
				tmpstr++;
				sprintf(ofile,"%s/crash.%d/%s",dumpdir, bounds,tmpstr);
				if ( (res=kl_cp_file(tfile,ofile)) > 0) {
					sprintf(dfile,"%-9s %s %s %lu %lu\n","module",tfile,tmpstr, res, sum_dump);
				} else {
					sprintf(dfile,"%-9s %s %s\n","module","failed to copy",tfile);
				}
			} else {
				sprintf(dfile,"%-9s %s %s %s\n","module",tfile,tmpstr,"cksum failed");
			}
		}
		write(indf, dfile, strlen (dfile));
		tmpstr=ptrstr+1;
	}

	if ( verbose_set == 1 ) {
		printf("Ok\n");
		savecrash_gentitle("Start retrieving dump",lentitle);
	 	if (local_debug) {
			fprintf(KL_ERRORFP,
				"%u pages out of %u\n",dh.dh_num_pages-dh.dh_num_notsaved,dh.dh_num_pages);
		}
		fflush(stdout);
	}

        /* move to the right offset in the device file */
        (void)lseek(devf, DUMP_PAGE_SZ + KL_PAGE_SIZE + offset, SEEK_SET);

	/* now calculate how many chunk to do */
	chunk_nb=0;
	
	if ( old_fashion_flag == 1 ) {
		/* so we need to calculate the size of a chunk */
		chunk_size_max=(dh.dh_memory_size*dh.dh_page_size)/chunk_max;
	} else {
		/* so we need to calculate the bitmap size */
		bitmap_max=(dh.dh_memory_size)/(8*chunk_max);
		bitmap32=(bitmap_max/32)+1;
		bitmap_size=bitmap32 * 32;
		bitmap_max=bitmap_size*8; /* this is the number of pages per image */
		/* so now we have the malloc for this bitmap, but now should align on 4k boundary  */
		bitmap_alloc=(((bitmap_max+7)/8+4095)/4096)*4096;
		/* take care that the last chunk will contain */
		/* fewer pages, so we have to calculate it.   */
		bitmap_last=(dh.dh_memory_size/8)-(bitmap_size*(chunk_max-1));
		bitmap_alloc_last=(((bitmap_last+7)/8+4095)/4096)*4096;
		bitmap=(uint8_t *)malloc(bitmap_alloc);
		memset(bitmap,0,bitmap_alloc);
	}
		
	chunk_size=0;
	chunk_new=0;
	loop=0;
	bitmap_chunk=0;
        
	/* now start rolling through the blocks */
        while (loop == 0) {
                /* try to read in the block */
		if (bool == 3) {  /* it means that it is the end of this device, need to look at the next one */
			close(devf);
			if ( verbose_set == 1 ) {
				printf("end of the device.\n");
				savecrash_gentitle("Look for next dumpdev",lentitle);
			}
			dumpdev_index++;
			i=0;
			while ( i == 0 ) {
				if ( dh.dh_dump_device[dumpdev_index] != 0 ) {
					if ( kl_read_dir(RootDev,dh.dh_dump_device[dumpdev_index]) == 1 ) {
						devf=kl_find_magic(Dev,buffer);
						if ( verbose_set == 1 ) {
							printf("%s\n",Dev);
							savecrash_gentitle("Retrieving dump",lentitle);
						}
						i=1;
					} else {
						fprintf(KL_ERRORFP,
							"Error: kl_dump_retrieve() can't open 0x%x\n",dh.dh_dump_device[dumpdev_index]);
					}
				} else {
					dumpdev_index++;
				}
				if ( dumpdev_index > MAX_DUMPDEV) {
					fprintf(KL_ERRORFP, "Warning: no more dumpdev!\n");
					close(devf);
					close(outf);
					kl_bounds_update(dumpdir, bounds + 1);
					return(-1);
				}
			}
			bool=0;
		}
		 	
		if (read(devf, (char *)buffer, DUMP_PAGE_SZ) < DUMP_PAGE_SZ ) {
                        fprintf(KL_ERRORFP, "Warning: short dump in dumpdev!\n");
                        close(devf);
                        close(outf);
                        close(indf);
                        kl_bounds_update(dumpdir, bounds + 1);
                        return (-1);
		}
	 	/* set toffset to the start of the buffer */
		toffset=0;
		bool=0;
		/* now we have our buffer loop to retrieve all pages */
		while ( bool == 0 ) {
			/* first, check if we don't achieve a chunk/image file */
			if ( old_fashion_flag == 1 ) {
				/* check if we are not above chunk_size_max */
				if ( chunk_size >= chunk_size_max ) {
			 		sum_dump=cksum(dfile);
					sprintf(tfile,"%-9s 0x%.16llx 0x%.16llx core.%d.%d %lu\n","chunk"
						,start_addr,(long long)chunk_size,chunk_nb,bounds,sum_dump);
					chunk_new=0;
					chunk_size=0;
					write(indf, tfile, strlen (tfile));
					if ( compression_flag == 1 ) {
						ret=kl_compress_module(dfile);
						if ( ret != 0 ) {
							return(-1);
						}
					}
					chunk_nb=chunk_nb+1;
					nb_physpages=0;
					addr_set=0;
					close(outf);
				}
			} else {
				if ( bitmap_chunk >= bitmap_max ) {
					/* we are at the end of an image, so we need to flush the bitmap */
					/* so go back to the start of the image. */
					if ( (ret=lseek(outf, 0, SEEK_SET)) < 0) {	
						fprintf(KL_ERRORFP, "Error: lseek() on image file failed: %d(%d)\n",errno,ret);
						close(devf);
						close(outf);
						close(indf);
						kl_bounds_update(dumpdir, bounds + 1);
						return (-1);
					}
					if (write(outf,(char *)bitmap,bitmap_alloc) != bitmap_alloc) {
						fprintf(KL_ERRORFP, "Warning: write() to dump file failed!\n");
						close(devf);
						close(outf);
						close(indf);
						kl_bounds_update(dumpdir, bounds + 1);
						return (-1);
					}
					sum_dump=cksum(dfile);
					sprintf(tfile,"%-9s %s.%d.%d 0x%.16x 0x%.16llx 0x%.16llx 0x%.16llx %lu\n","image"
						,"image",bounds,chunk_nb,0,(long long)chunk_size,start_page,end_page,sum_dump);
					chunk_new=0;
					chunk_size=0;
					write(indf, tfile, strlen (tfile));
					if ( compression_flag == 1 ) {
						ret=kl_compress_module(dfile);
						if ( ret != 0 ) {
							return(-1);
						}
					}
					chunk_nb=chunk_nb+1;
					if ( chunk_nb >= (chunk_max-1) ) {
						bitmap_alloc=bitmap_alloc_last;
						bitmap_max=(bitmap_alloc*8)+1;
					}
					bitmap_chunk=0;
					memset(bitmap,0,bitmap_alloc);
					addr_set=0;
					close(outf);
				}
			}

			/* then retrieve the dump_page header */
			memcpy((char *)&dp, (buffer + toffset), sizeof(dp));
			toffset += sizeof(dp);
			if(local_debug) {	
                       		fprintf(KL_ERRORFP,
                                	"dp.dp_address = 0x%"FMT64"x, "
                                	"dp.dp_flags = 0x%x, dp.dp_size = %d\n ",
                                	dp.dp_address, dp.dp_flags, dp.dp_size);
			}
                	/* sanity check */
                	if ((!dp.dp_flags) || ((!dp.dp_size) && (!(dp.dp_flags & DUMP_DH_END || dp.dp_flags & DUMP_DH_FREE))) || (dp.dp_size > dh.dh_page_size)) {
                                	fprintf(KL_ERRORFP,
                                        	"Error: dump flags in dumpdev page index invalid!\n");
                                	close(devf);
                                	close(outf);
                        		close(indf);
                                	kl_bounds_update(dumpdir, bounds + 1);
                                	return (-1);
                	}

                	/* make sure we aren't at the end of the dump */
                	if (dp.dp_flags & DUMP_DH_END) {
				bool=1;
			}

			/* make sure we re not at the end of DUMP_PAGE_SZ */
			if (dp.dp_flags & DUMP_DH_END_BLOCK) {
				bool=2;
			}

			/* make sure that we are not at the end of the device */
			if (dp.dp_flags & DUMP_DH_END_DEV) {
				bool=3;
			}
                	
			/* read in the dump page itself */
			if ( dp.dp_size != 0) {
				memcpy((char *)pagebuf, (buffer + toffset), dp.dp_size);
				toffset += dp.dp_size;
			} else if ( old_fashion_flag == 1 ) {
				memset((char *)pagebuf,0,dh.dh_page_size);
			} 

			/* need to setup start_addr if not set */
			if ( addr_set == 0 ) {
				start_addr=dp.dp_address;
				start_page=dp.dp_address/dh.dh_page_size;
				addr_set=1;
			}

			/* if it is a compressed page, need to uncompress */
			/* actually don't support page_size != KL_PAGE_SIZE */
                	if ( dp.dp_flags & DUMP_DH_COMPRESSED ) {
				switch(dh.dh_dump_compress) {
                			case DUMP_COMPRESS_NONE:
						fprintf(KL_ERRORFP, "Error: page is compressed, but DUMP_COMPRESS_NONE\n");
						page_size=0;
                        			break;
                			case DUMP_COMPRESS_RLE:
                        			page_size=kl_uncompress_rle_page((unsigned char*)pagebuf,
							(unsigned char*)uncompr_page, dp.dp_size);
                        			break;
                			case DUMP_COMPRESS_GZIP:
						fprintf(KL_ERRORFP, "Error: gzip not supported.\n");
						page_size=0;   
                        			break;
                			case DUMP_COMPRESS_LBZ2:
						page_size=kl_uncompress_lbz2_page((unsigned char*)pagebuf,
							(unsigned char*)uncompr_page, dp.dp_size);
                        			break;
                			default:
						fprintf(KL_ERRORFP, "Error: unknown compression.\n");
						page_size=0;
        			}
                        	if ( page_size != KL_PAGE_SIZE) {
					fprintf(KL_ERRORFP, "Error: kl_uncompress_page() failed!\n");
                        		close(devf);
                        		close(outf);
                        		close(indf);
                        		kl_bounds_update(dumpdir, bounds + 1);
                        		return (-1);
				}
				ptr_page=uncompr_page;
			} else {
				ptr_page=pagebuf;
				page_size=dp.dp_size;
                	} 

			if ( bool != 1 ) {
				if ( (end_page) != ((dp.dp_address/dh.dh_page_size)-1) && end_page != 0) {
					fprintf(KL_ERRORFP,
						"ERROR: pages missing, prev: 0x%llx next:0x%llx\n",
						end_page,dp.dp_address/dh.dh_page_size);
					close(devf);
					close(outf);
					close(indf);
					kl_bounds_update(dumpdir, bounds + 1);
					return (-1);
				 }	
				chunk_size=chunk_size+page_size;
				end_page=dp.dp_address/dh.dh_page_size;
			}

			if ( old_fashion_flag != 1 ) {	
				/* need to calculate the bitmap value */
				/* first calculate the bytes */
				bytes_in_bitmap=bitmap[bitmap_chunk/8];
				bytes_in_bitmap=bytes_in_bitmap >> 1; /* right shift by one first */
				if ( !(dp.dp_flags & DUMP_DH_FREE) ) {
					bytes_in_bitmap=bytes_in_bitmap+0x80; /* then add 0x1000000 */
				}
				bitmap[bitmap_chunk/8]=bytes_in_bitmap;
				bitmap_chunk++;
			}
		
			/* need to count the number of pages */
			/* so -> page_size/dh_page_size      */
		 	nb_physpages=nb_physpages+(page_size/dh.dh_page_size);

                	/* try to write in the dump page itself */
			/* check if it is a new chunk, if yes need create the file */
			/* make the new filename */
			if ( chunk_new == 0 && bool !=1 ) {
				if ( old_fashion_flag == 1 ) {
					sprintf(dfile, "%s/crash.%d/core.%d.%d", dumpdir, bounds, chunk_nb, bounds);
				} else {
					sprintf(dfile,"%s/crash.%d/image.%d.%d", dumpdir, bounds, bounds, chunk_nb);
				}
				if ((outf = open(dfile, O_CREAT|O_RDWR|O_TRUNC, (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
					fprintf(KL_ERRORFP, "Error: open() of dump file \"%s\" failed!\n", dfile);
					close(devf);
                			return (-1);
				}
				if ( old_fashion_flag != 1 ) {
					/* because it is a bitmap with pages after the bitmap, need */
					/* to flush at least the first bitmap. */
					if (write(outf,(char *)bitmap,bitmap_alloc) != bitmap_alloc) {
						fprintf(KL_ERRORFP, "Warning: write() to dump file failed!\n");
						close(devf);
                        			close(outf);
                        			close(indf);
                        			kl_bounds_update(dumpdir, bounds + 1);
                        			return (-1);
						}
					chunk_size=chunk_size+bitmap_alloc;
				}
				/* and we have all page information so set the addresse */
				/* of the first page and last pages in this image.      */
				chunk_new = 1;
			}
               		if ( old_fashion_flag == 1 && page_size== 0 ) {
				page_size=dh.dh_page_size;
				chunk_size=chunk_size+page_size;
			}
			if ( bool != 1 ) { 
				if (write(outf, (char *)ptr_page, page_size) != page_size) {
                        		/* if this fails, we may have run out of space */
                        		fprintf(KL_ERRORFP, "Warning: write() to dump file failed!\n");
                        		close(devf);
                        		close(outf);
                        		close(indf);
                        		kl_bounds_update(dumpdir, bounds + 1);
                        		return (-1);
                		}
			}
        	}
		if ( bool == 1 ) {
			if (lseek(devf, KL_PAGE_SIZE + offset, SEEK_SET) < 0) {
			fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
                                close(devf);
                                close(outf);
                                close(indf);
                                kl_bounds_update(dumpdir, bounds + 1);
                                return (-1);
                        }
			/* then change the flag and write on the header */
                        dh.dh_dump_flags=dh.dh_dump_flags+DUMP_FLAGS_SAVED;
                        if ((write(devf, (char *)&dh, sizeof(dump_header_t))) != sizeof(dump_header_t)) {
                                fprintf(KL_ERRORFP, "Error: write() on dumpdev failed!\n");
                                close(devf);
                                close(outf);
                                close(indf);
                                kl_bounds_update(dumpdir, bounds + 1);
                                return (-1);
                        }
                        close(devf);
			loop=1;
		}
	}
 	/* if we reach this point it means that the dump is */
	/* complete, so need to cksum our last chunk        */	
	if ( old_fashion_flag != 1 ) {
		if ( (ret=lseek(outf, 0, SEEK_SET)) < 0) {
			fprintf(KL_ERRORFP, "Error: final lseek() on image file failed: %d(%d)\n",errno,ret);
			close(devf);
			close(outf);
			close(indf);
			kl_bounds_update(dumpdir, bounds + 1);
			return (-1);
		}
		if (write(outf,(char *)bitmap,bitmap_alloc) != bitmap_alloc) {
			fprintf(KL_ERRORFP, "Warning: write() to dump file failed!\n");
			close(devf);
			close(outf);
			close(indf);
			kl_bounds_update(dumpdir, bounds + 1);
			return (-1);
		}
		sum_dump=cksum(dfile);
		sprintf(tfile,"%-9s %s.%d.%d 0x%.16x 0x%.16llx 0x%.16llx 0x%.16llx %lu\n","image"
			,"image",bounds,chunk_nb,0,(long long)chunk_size,start_page,end_page,sum_dump);
		write(indf, tfile, strlen (tfile));
		close(indf);
	}

	if ( compression_flag == 1 ) {
		ret=kl_compress_module(dfile);
		if ( ret != 0 ) {
			return(-1);
		}
	}
	if ( verbose_set == 1 ) {
		printf("Ok\n");
		fflush(stdout);
	}
	kl_bounds_update(dumpdir, bounds + 1);
	close(outf);
	return(loop);
}

/*
 * check_type_dir() -- run the stat() operation without having all the
 *                     inline definitions conflicted by the kernel.  Make
 *                     sure <path> is a directory.
 */
int
check_type_dir(char *path)
{
        struct stat sbuf;
	int ret=0;

	if ( verbose_set == 1 ) {
		savecrash_gentitle("Checking directory",lentitle);
		fflush(stdout);
	}
        if (stat(path, &sbuf) < 0) {
                return (1);
        }
	if ( S_ISDIR(sbuf.st_mode) ) {
                if ( verbose_set == 1 ) {
			printf("OK\n");
		}
                ret=0;
        } else {
                printf("ERROR: can't open dump directory.\n");
                ret=1;
        }     
        return (ret);
}

/*
 * Name: kl_get_bounds()
 * Func: Get the bounds number from the bounds file located in <dumpdir>
 *       Returns the bounds number, or 0 if no bounds file exists.
 */
int
kl_bounds_retrieve(char *dumpdir)
{
        int dirf, bounds;
        char *tfile;

	if ( verbose_set == 1 ) {
		savecrash_gentitle("Read bound",lentitle);
		fflush(stdout);
	}
        tfile = (char *)malloc(strlen(dumpdir) + 10);
        if (!tfile) {
                return (0);
        }

        sprintf(tfile, "%s/bounds", dumpdir);
        if ((dirf = open(tfile, O_RDONLY, 0)) < 0) {
		if ( verbose_set == 1 ) {
			printf("No bound: 0\n");
		}
                return (0);
        }

        (void)read(dirf, tfile, strlen(tfile));
        bounds = atoi(tfile);
        close(dirf);
	if ( verbose_set == 1 ) {
		printf("%d\n",bounds);
	}
	return (bounds);
}

/*
 * Name: kl_bounds_update()
 * Func: Update the bounds number with a new value.  The function takes
 *       the <dumpdir> along with the new bounds number.  Returns 0 for
 *       failure, 1 for success.
 */
void
kl_bounds_update(char *dumpdir, int bounds)
{
        int dirf;
        char *tfile;

        tfile = (char *)malloc(strlen(dumpdir) + 10);
        if (!tfile) {
                return;
        }

        sprintf(tfile, "%s/bounds", dumpdir);
        if ((dirf = open(tfile, O_RDWR|O_CREAT,
                (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
                        fprintf(KL_ERRORFP, "Warning: could not "
                                "create new bounds file in %s!\n", dumpdir);
                        return;
        }
	sprintf(tfile, "%d\n", bounds);
        if (write(dirf, tfile, strlen(tfile)) != strlen(tfile)) {
                fprintf(KL_ERRORFP, "Warning: could not "
                        "write to bounds file in %s!\n", dumpdir);
                close(dirf);
                return;
        }
        close(dirf);
        return;
}

unsigned long kl_cp_file(char *src,char *dest)
{
	int infile, outfile, loop, mod_size=0;
	char buf[8192];
	size_t count;

	/* open src file */
	if (( infile = open(src, O_RDONLY, 0)) < 0) {
		fprintf(KL_ERRORFP, "Error: open() on %s failed!\n",src);
		return (-1);
	}

	/* open dest file */
	if (( outfile = open(dest, O_CREAT|O_RDWR|O_TRUNC,(S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
		fprintf(KL_ERRORFP, "Error: open() on %s failed!\n",src);
		return (-1);
	}
	loop=0;
	while (loop == 0) {
		count=read(infile , buf, sizeof(buf));
		if ( count == 0 ) {
			loop=1;
		}
		if ( write(outfile, buf, count) != count) {
			loop=1;
			fprintf(KL_ERRORFP, "Error: write() on %s failed!\n",dest);
			return (-1);
		}
		mod_size=mod_size+count;
	}

	if ( compression_flag == 1 ) {
		loop=kl_compress_module(dest);
		if ( loop != 0 ) {
			return(mod_size);
		}
	}
	return(mod_size);
}

int
kl_find_module(char *mod)
{
  char buf[KL_PAGE_SIZE];
  char *ptrav;
  FILE *fic;

	sprintf(buf,"insmod -n -q %s 2>/dev/null",mod);
	if ( (fic=popen(buf, "r")) != NULL ) {
		fgets(buf, sizeof(buf), fic);
      		pclose(fic);
      		if ((ptrav=strchr(buf, ' ')) != NULL) {
        		ptrav++;
			ptrav[strlen(ptrav)-1]='\0';
			strncpy(mod,ptrav,KL_PAGE_SIZE);
      		} else {
			buf[0]='\0';
			fprintf(KL_ERRORFP, "Error: insmod -n -q %s failed!\n",mod);
			return(1);
		}
	} else {
		fprintf(KL_ERRORFP, "Error: insmod -n -q %s failed!\n",mod);
		return(1);
	}
	return(0);
}

int kl_compress_module(char *mod)
{
  char buf[KL_PAGE_SIZE];
  int ret;

	sprintf(buf,"gzip %s",mod);
	if ( (ret=system(buf)) != 0) {
		fprintf(KL_ERRORFP, "Error: gzip() on %s failed!\n",mod);
	}
	return(ret);
}

int kl_uncompress_lbz2_page(unsigned char *cbuf, unsigned char *ucbuf, uint32_t blk_size)
{
	if (lbz2_init_flag == 0 ) {
		bzip_init(&map_lbz2_decode);
		lbz2_init_flag=1;
	}
	return((int)bzip_uncompressBlock(cbuf,ucbuf,blk_size,KL_PAGE_SIZE,0));
}
		
/*
 * kl_uncompress_page() -- Uncompress a buffer of data
 *
 * Return the size, otherwise 0.
 */

int 
kl_uncompress_rle_page(unsigned char *cbuf, unsigned char *ucbuf, uint32_t blk_size)
{
	int i;
	unsigned char value, count, cur_byte;
	kaddr_t ri, wi;
	
	/* initialize the read / write indices */ 
	ri = wi = 0;
	
	/* otherwise decompress using run length encoding */
	while(ri < blk_size) {
		cur_byte = cbuf[ri++];
		if (cur_byte == 0) {
			count = cbuf[ri++];
			if (count == 0) {
				ucbuf[wi++] = 0;
			} else {
				value = cbuf[ri++];
				for (i = 0; i <= count; i++) {
					ucbuf[wi++] = value;
				}
			}
		} else {
			ucbuf[wi++] = cur_byte;
		}

		/* if our write index is beyond the page size, exit out */
		if (wi > KL_PAGE_SIZE) {
			fprintf(KL_ERRORFP,
				"ERROR: Attempted to decompress beyond page boundaries: buffer corrupted!\n");
			return (0);
		}
	}
	/* set return size to be equal to uncompressed size (in bytes) */
	return (int)wi;
}

int kl_read_dir(char *dir,int dev)
{
        DIR           *dirp;
        struct dirent *ficp;
        struct stat   ficstat;
        char          buf[4096];

        if ((dirp=opendir(dir)) == (DIR *)NULL) {
                return(0);
        }
        while ((ficp=readdir(dirp)) != NULL) {
                if ((!strcmp(ficp->d_name, ".")) || (!strcmp(ficp->d_name, ".."))) {
                        continue;
                }
                sprintf(buf, "%s/%s", dir, ficp->d_name);
                if (lstat(buf, &ficstat) != 0) {
                        printf("error: %s lstat() failed.\n",buf);
                        continue;
                }
                if ( (ficstat.st_mode & S_IFLNK ) == S_IFLNK ) {
                        continue;
                }
                if (S_ISDIR(ficstat.st_mode)) {
                        if (kl_read_dir(buf,dev) != 1 ) {
                        	continue;
			} else {
				return(1);
			}
                }

                if (S_ISCHR(ficstat.st_mode) || S_ISBLK(ficstat.st_mode)) {
                        if ( lstat(buf, &ficstat) != 0) {
                                printf("error: %s stat() failed.\n",buf);
                                continue;
                        }
                        if ( ficstat.st_rdev == dev ) {
                                sprintf(Dev,"%s",buf);
				return(1);
                        }
                }
        }
        closedir(dirp);
	return(0);
}

int kl_find_magic(char *dump_dev,char *buffer)
{
	int	devf;
	char	msg[1024];

	if ( verbose_set == 1 ) {
		sprintf(msg,"Open dump device: %s",dump_dev);	
		savecrash_gentitle(msg,lentitle);
		fflush(stdout);
	}
	/* try to open the output file */
	if ((devf = open(dump_dev, O_RDWR, 0)) < 0) {
		fprintf(KL_ERRORFP, "Error: open() on dumpdev %s failed!\n",dump_dev);
		return (-1);
	}
	if ( verbose_set == 1 ) {
		printf("Ok\n");
		savecrash_gentitle("Read dump header",lentitle);
		fflush(stdout);
	}
	/* move beyond the swap header */
	offset=0;
	dh.dh_magic_number=0;
	while ( dh.dh_magic_number != DUMP_MAGIC_NUMBER ) {
		if (lseek(devf, KL_PAGE_SIZE + offset , SEEK_SET) < 0) {
			fprintf(KL_ERRORFP, "Error: lseek() on dumpdev %s failed!\n",dump_dev);
			close(devf);
			return (-1);
		}
		/* try to read the dump header */
		/* first check the size of both header base and arch specifique */
		if (read(devf, (char *)buffer, DUMP_PAGE_SZ) < DUMP_PAGE_SZ) {
			fprintf(KL_ERRORFP, "Error: read() of dump header failed!\n");
			close(devf);
			return (-1);
		}
		memcpy((char *)&dh,buffer,sizeof(dump_header_t));
		memcpy((char *)&dha,(buffer + sizeof(dump_header_t)),sizeof(dump_header_asm_t));
		/* validate the dump header */
		if (( offset > DEVBLKSIZE )) {
			fprintf(KL_ERRORFP,"Error: DUMP_MAGIC_NUMBER in dump header invalid! 0x%x\n",KL_PAGE_SIZE);
			close(devf);
			return (-1);
		}
		if ( dh.dh_magic_number != DUMP_MAGIC_NUMBER ) {
			offset++;
		}
	}
	/* check if the size of the header is the same (pb between 64/32 bits OS) */
	if ( dh.dh_header_size != sizeof(dump_header_t) ) {
		fprintf(KL_ERRORFP,"Error: headers sizes are not equal: dump: %d binary: %d\n"
			,dh.dh_header_size,sizeof(dump_header_t));
		close(devf);
		return (-1);
	}

	/* check if dump already saved */
	if (dh.dh_dump_flags & DUMP_FLAGS_SAVED) {
		if ( resave_set == 0 ) {
			fprintf(KL_ERRORFP,
				"Error: DUMP_FLAGS_SAVED set!\n");
			close(devf);
			return(-1);
		}
	} 
	
        /* validate the architecture-specific dump header */
        if (dha.dha_magic_number != DUMP_ASM_MAGIC_NUMBER) {
                        fprintf(KL_ERRORFP,
                                "Error: DUMP_ASM_MAGIC_NUMBER in "
                                "dump arch header invalid: 0x%llx!\n",dha.dha_magic_number);
                close(devf);
                return (-1);
        }
	return devf;
}
		
