/*
 * Copyright 1999 Silicon Graphics, Inc. All rights reserved.
 */

#ifndef __KL_MEM_H
#define __KL_MEM_H

#include "asm/kl_mem.h"

/* CORE_TYPE indicates type of dump
 */
typedef enum {
	dev_kmem,   /* image of /dev/kmem, a running kernel */
	reg_core,   /* Regular (uncompressed) core file */
	s390_core,  /* s390 core file */
	cmp_core    /* compressed core file */
} CORE_TYPE;

/* Information about vmcore file (core dump or live system memory)
 */
typedef struct meminfo_s {
	CORE_TYPE	core_type; /* type of core file   */
	char	       *dump;  /* pathname for dump  */
	char	       *map;  /* pathname for map file */
	int		core_fd;   /* file descriptor for dump file  */
	int		ptrsz;     /* Number of bytes in a pointer type */
	int		rw_flag;   /* O_RDONLY/O_RDWR (/dev/kmem only) */
} meminfo_t;

/**
 ** Function prototypes
 **/
void kl_free_meminfo(
	meminfo_t *	/* pointer to meminfo_s struct */);

meminfo_t *kl_init_meminfo(
	char *		/* map file */, 
	char *		/* dump */, 
	int 		/* rwflag */);

uint64_t kl_readmem(
	kaddr_t 	/* physical address to start reading from */, 
	unsigned 	/* number of bytes to read */, 
	void *	 	/* pointer to buffer */);

int kl_virtop(
	kaddr_t 	/* virtual address to translate */, 
	void *	 	/* pointer to mem_map for address translation */,
	kaddr_t *       /* pointer to physical address to return */);

uint64_t get_block(
	kaddr_t 	/* virtual address */, 
	unsigned 	/* size of block to read in */, 
	void *		/* pointer to buffer */,		
	void * 		/* pointer to mmap */); 

/* Wrapper that eliminates the mmap parameter
 */
#define GET_BLOCK(a, s, b) get_block(a, s, (void *)b, (void *)0)

uint64_t kl_get_kaddr(
	kaddr_t 	/* virtual address */, 
	void * 		/* pointer to buffer */); 

kaddr_t kl_kaddr_to_ptr(
	kaddr_t		/* kernel virtual address */);

uint64_t kl_uint(
	void *	 	/* pointer to buffer containing struct */, 
	char *		/* name of struct */, 
	char *		/* name of member */, 
	unsigned 	/* offset */);

int64_t kl_int(
	void *	 	/* pointer to buffer containing struct */, 
	char *		/* name of struct */, 
	char *		/* name of member */, 
	unsigned 	/* offset */);

kaddr_t kl_kaddr(
	void *	 	/* pointer to buffer containing struct */, 
	char *		/* name of struct */, 
	char *		/* name of member */); 

kaddr_t kl_reg(void *p, char *s, char *m);

/* functions defined in arch/xxx/kl_page.c */
kaddr_t pgd_offset(
	kaddr_t         /* kernel virtual address of page directory */,
	kaddr_t         /* kernel virtual address */);

kaddr_t pmd_offset(
	kaddr_t         /* kernel virtual address of page middle directory */,
	kaddr_t         /* kernel virtual address */);

kaddr_t pte_offset(
	kaddr_t         /* kernel virtual address of page table */,
	kaddr_t         /* kernel virtual address */);

kaddr_t mmap_virtop(
	kaddr_t vaddr   /* kernel virtual address */,
	void *mmp       /* pointer to struct mm_struct */);

#define KL_KADDR_IS_PHYSICAL(vaddr) ((vaddr > KL_PAGE_OFFSET) && \
                                     (vaddr < KL_HIGH_MEMORY))

#endif /* __KL_MEM_H */
