/*
 * crashconf.c -- dump configuration utility based on lkcd_util
 * Created by: Matt D. Robinson (yakker@aparity.com)
 * Copyright 2001 Matt D. Robinson (yakker@aparity.com), all rights reserved.
 * 2002 modified by Bruno Vidal (bruno_vidal@hp.com).
 * This source code is released under the terms of version 2 of the GNU GPL.
 */

static char *SccsId = "@(#)crashconf $Revision: 2.1 $  $Date: 2003/03/30 bruno_vidal@hp.com\n";

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <crashtool.h>
#include <linux/module.h>
#include <linux/dump.h>

#define DUMP_DEVICE	"/dev/dump"
#define DUMP_TRUE	1
#define DUMP_FALSE	0

/*
 * Name: main()
 * Func: A quick-and-dirty dump configuration tool.  Should suffice
 *       for the time being.
 */
int
main(int argc, char **argv)
{
	long dnum = 0;
	int dfd, fd, err, compress, level, flags, device_num, c;
	int device_name_set = DUMP_FALSE, level_set = DUMP_FALSE;
	int flags_set = DUMP_FALSE, compress_set = DUMP_FALSE;
	int device_num_set = DUMP_FALSE, query_set = DUMP_FALSE;
	int version_set = DUMP_FALSE, device_name_remove = DUMP_FALSE;
	int device_num_remove = DUMP_FALSE, index=0, num_pages=0;
	char *device_name, tbuf[1024];
	struct stat stbuf;
        char *base;  

        base=basename(argv[0]);


	/* check for root */
	if (getuid() != 0) {
		fprintf(stderr, "Error: You must run this program as root!\n");
		return (-1);
	}

	/* walk through the options */
	compress = level = flags = device_num = 0;
	while ((c = getopt(argc, argv, "qc:d:D:f:l:r:R:V")) > 0) {
		switch (c) {
			case 'c':
				if (compress_set == DUMP_TRUE) {
					goto usage;
				}
				compress_set = DUMP_TRUE;
				compress = atoi(optarg);
				break;

			case 'd':
				if ((device_name_set == DUMP_TRUE) ||
					(device_num_set == DUMP_TRUE)) {
						goto usage;
				}
				device_name_set = DUMP_TRUE;
				device_name = optarg;
				break;

			case 'r':
				if ((device_name_set == DUMP_TRUE) || (device_name_remove == DUMP_TRUE) ||
					(device_num_set == DUMP_TRUE) || (device_num_remove == DUMP_TRUE)) {
						goto usage;
				}
				device_name_remove = DUMP_TRUE;
				device_name = optarg;
				break;

			case 'R':
				if ((device_name_set == DUMP_TRUE) || (device_name_remove == DUMP_TRUE) ||
					(device_num_set == DUMP_TRUE) || (device_num_remove == DUMP_TRUE)) {
						goto usage;
				}
				device_num_remove = DUMP_TRUE;
				device_num = atoi(optarg);
				break;

			case 'D':
				if ((device_name_set == DUMP_TRUE) ||
					(device_num_set == DUMP_TRUE)) {
						goto usage;
				}
				device_num_set = DUMP_TRUE;
				device_num = atoi(optarg);
				break;

			case 'f':
				if (flags_set == DUMP_TRUE) {
					goto usage;
				}
				flags_set = DUMP_TRUE;
				flags = atoi(optarg);
				break;

			case 'l':
				if (level_set == DUMP_TRUE) {
					goto usage;
				}
				level_set = DUMP_TRUE;
				level = atoi(optarg);
				break;

			case 'q':
				if (query_set == DUMP_TRUE) {
					goto usage;
				}
				query_set = DUMP_TRUE;
				break;

			case 'V':
                                if (version_set == DUMP_TRUE) {
                                        goto usage;
                                }
                                version_set = DUMP_TRUE;
                                break;

			default:
				goto usage;
		}
	}

	/* first, write version if needed */
	if (version_set == DUMP_TRUE) {
		printf("Version: %s",SccsId);
	}

	/* make sure -q isn't specified with anything else */
	if (query_set == DUMP_TRUE) {
		if ((device_name_set == DUMP_TRUE) ||
			(device_num_set == DUMP_TRUE) ||
			(level_set == DUMP_TRUE) ||
			(flags_set == DUMP_TRUE) ||
			(compress_set == DUMP_TRUE)) {
				goto usage;
		}

		/* open the dump device for queries */
		if ((dfd = open(DUMP_DEVICE, O_RDONLY)) < 0) {
			perror("open of dump device");
			return (dfd);
		}

		/* get dump compression */
		if ((compress = ioctl(dfd, DIOGDUMPCOMPRESS, NULL)) < 0) {
			perror("ioctl() query for dump compression failed");
			close(dfd);
			return (err);
		}

		/* get dump flags */
		if ((flags = ioctl(dfd, DIOGDUMPFLAGS, NULL)) < 0) {
			perror("ioctl() query for dump flags failed");
			close(dfd);
			return (flags);
		}

		/* get dump level */
		if ((level = ioctl(dfd, DIOGDUMPLEVEL, NULL)) < 0) {
			perror("ioctl() query for dump level failed");
			close(dfd);
			return (level);
		}

		/* get the number of pages */
		if (( num_pages = ioctl(dfd, DIAGDUMPNUMPAGES, NULL)) < 0) {
			perror("ioctl() query for num pages failed.\n");
			close(dfd);
			return (num_pages);
		}
		printf("Number of pages to save: %10s %d\n","",num_pages);

		/* get the number of configured device */
		if ((index = ioctl(dfd, DIOGDINDEX, NULL)) < 0) {
			perror("ioctl() query for dump index failed");
			close(dfd);
			return (index);
		}

		printf("Number of configured device: %6s %d\n","",(int)index);

		/* get device to dump to (if specified) */
		for (c = 0;c < index;c++) {
			dnum=c;
			if ((err = ioctl(dfd, DIOGDUMPDEV, dnum)) < 0) {
				perror("ioctl() for dump device failed");
				close(dfd);
				return (err);
			}
			printf("Configured dump device: %11s 0x%lx\n","", (long)err);
		}
	
		memset(tbuf, 0, 1024);
		if (flags == DUMP_FLAGS_NONE) {
			strcat(tbuf, "DUMP_FLAGS_NONE|");
		} else if (flags & DUMP_FLAGS_NONDISRUPT) {
			strcat(tbuf, "DUMP_FLAGS_NONDISRUPT|");
		}
		if (tbuf[strlen(tbuf)-1] == '|') {
			tbuf[strlen(tbuf)-1] = 0;
		}
		printf("Configured dump flags: %12s %s\n","", tbuf);

		memset(tbuf, 0, 1024);
		if (level == DUMP_LEVEL_NONE) {
			strcat(tbuf, "DUMP_LEVEL_NONE|");
		}
		if (level & DUMP_LEVEL_HEADER) {
			strcat(tbuf, "DUMP_LEVEL_HEADER|");
		}
		if (level & DUMP_LEVEL_KERN) {
			strcat(tbuf, "DUMP_LEVEL_KERN|");
		}
		if (level & DUMP_LEVEL_USED) {
			strcat(tbuf, "DUMP_LEVEL_USED|");
		}
		if (level & DUMP_LEVEL_ALL) {
			strcat(tbuf, "DUMP_LEVEL_ALL|");
		}
		if (tbuf[strlen(tbuf)-1] == '|') {
			tbuf[strlen(tbuf)-1] = 0;
		}
		printf("Configured dump level: %12s %s\n","", tbuf);

		memset(tbuf, 0, 1024);
		if (compress == DUMP_COMPRESS_NONE) {
			strcat(tbuf, "DUMP_COMPRESS_NONE|");
		}
		if (compress & DUMP_COMPRESS_RLE) {
			strcat(tbuf, "DUMP_COMPRESS_RLE|");
		}
		if (compress & DUMP_COMPRESS_GZIP) {
			strcat(tbuf, "DUMP_COMPRESS_GZIP|");
		}
		if (compress & DUMP_COMPRESS_LBZ2) {
			strcat(tbuf, "DUMP_COMPRESS_LBZ2|");
		}
		if (tbuf[strlen(tbuf)-1] == '|') {
			tbuf[strlen(tbuf)-1] = 0;
		}
		printf("Configured dump compression method: %s\n", tbuf);

		/* bail out */
		close(dfd);
		return (0);
	} else if ((device_name_set != DUMP_TRUE) &&
		(device_num_set != DUMP_TRUE) && (level_set != DUMP_TRUE) &&
		(flags_set != DUMP_TRUE) && (compress_set != DUMP_TRUE) &&
		(device_name_remove != DUMP_TRUE) && (device_num_remove != DUMP_TRUE)) {
			/* make sure _something_ is set */
			goto usage;
	}


	/* open the dump device for setting dump fields */
	if ((dfd = open(DUMP_DEVICE, O_RDWR)) < 0) {
		perror("open of dump device");
		return (dfd);
	}

	/* set the dnum -- dnum will != 0 after */
	if ((device_name_set == DUMP_TRUE) || (device_num_set == DUMP_TRUE)) {
		if (device_name_set == DUMP_TRUE) {
			if ((fd = open(device_name, O_RDONLY)) < 0) {
				perror("open of device name");
				close(dfd);
				return (fd);
			}

			if ((err = fstat(fd, &stbuf)) < 0) {
				perror("fstat of device name");
				close(dfd);
				close(fd);
				return (err);
			}
			dnum = stbuf.st_rdev;
			close(fd);
		} else {
			if (device_num <= 0) {
				fprintf(stderr, "Error: -d value must be > 0!\n");
				close(dfd);
				return (-1);
			}
			dnum = (dev_t)device_num;
		}
	}

	/* remove a configured device */
	if ((device_name_remove == DUMP_TRUE) || (device_num_remove == DUMP_TRUE)) {
		if (device_name_remove == DUMP_TRUE) {
			if ((fd = open(device_name, O_RDONLY)) < 0) {
				perror("open of device name");
				close(dfd);
				return (fd);
			}

			if ((err = fstat(fd, &stbuf)) < 0) {
				perror("fstat of device name");
				close(dfd);
				close(fd);
				return (err);
			}
			dnum = stbuf.st_rdev;
			close(fd);
		} else {
			if (device_num <= 0) {
				fprintf(stderr, "Error: -d value must be > 0!\n");
				close(dfd);
				return (-1);
			}
			dnum = (dev_t)device_num;
		}
	}
			

	/* set dump compression */
	if (compress_set == DUMP_TRUE) {
		if ((err = ioctl(dfd, DIOSDUMPCOMPRESS, compress)) < 0) {
			perror("ioctl() for dump compression failed");
			close(dfd);
			return (err);
		}
	}

	/* set dump flags */
	if (flags_set == DUMP_TRUE) {
		if ((err = ioctl(dfd, DIOSDUMPFLAGS, flags)) < 0) {
			perror("ioctl() for dump flags failed");
			close(dfd);
			return (err);
		}
		printf("Num pages: %d\n",err);
	}

	/* set dump level */
	if (level_set == DUMP_TRUE) {
		if ((err = ioctl(dfd, DIOSDUMPLEVEL, level)) < 0) {
			perror("ioctl() for dump level failed");
			close(dfd);
			return (err);
		}
	}

	/* set device to dump to (if specified) */
	if (dnum != (dev_t)0) {
		if (device_name_set == DUMP_TRUE) { 
			printf("Configured dump device: 0x%lx\n", (long)dnum);
			if ((err = ioctl(dfd, DIOSDUMPDEV, dnum)) < 0) {
				perror("ioctl() for dump device failed");
				close(dfd);
				return (err);
			}
		} else {
			printf("Remove dump device: 0x%lx\n", (long)dnum);
			if ((err = ioctl(dfd, DIORDUMPDEV, dnum)) < 0) {
				perror("ioctl() for dump device failed");
				close(dfd);
				return (err);
			}
		}
	}

	return (0);

usage:
	printf("Usage: %s [-d device_name | -D device_num] [-c compression_type] [-f flags] [-l level] {-V}\n",base);
	printf("Usage: %s [-r device_name | -R device_num] {-V}\n",base);
	printf("Usage: %s [-q] {-V}\n", base);
	printf("\t-d : device name where to dump to.\n");
	printf("\t-D : device number (like 0x804 for /dev/sda4).\n");
	printf("\t-c : compression type (0,1,2,4).\n");
	printf("\t-f : flag (see man page).\n");
	printf("\t-l : dump level (full, header only, etc..).\n");
	printf("\t-V : version.\n");
	printf("\t-r : remove a configured dump device using device name.\n");
	printf("\t-R : remove a configured dump device using device number.\n");
	printf("\t-q : query option.\n");
	return (1);
}
